#pragma once

#include <macs/label.h>
#include <map>

namespace macs {

class LabelsRepository;

class LabelSet : public std::map<std::string, Label> {
    static const Label & label(const value_type & v) { return v.second; }
    static const Label & label(const_iterator i) { return label(*i); }
public:
    typedef std::map<std::string, Label> Base;
    typedef Label::Symbol Symbol;
    typedef Label::Type Type;
    typedef std::string Name;

    using Base::find;

    const_iterator find(const Symbol & symbol) const {
        return std::find_if(begin(), end(), [&symbol](const value_type &v) {
            return label(v).symbolicName() == symbol;
        } );
    }

    const_iterator find(const Name & name, const Type & type) const {
        return std::find_if(begin(), end(), [&](const value_type &v) {
                return label(v).name() == name && label(v).type() == type;
            } );
    }

    const Label & at(const Lid & lid) const {
        const auto i = find(lid);
        if (i == end()) {
            throw system_error(error_code(error::noSuchLabel),
                    "LabelSet::at: no lid '" + lid + "'");
        }
        return label(i);
    }

    const Label & at(const Symbol & symbol) const {
        const auto i = find(symbol);
        if (i == end()) {
            throw system_error(error_code(error::noSuchLabel),
                    "LabelSet::at: no symbol '" + symbol.title() + "'");
        }
        return label(i);
    }

    const Label & at(const Name & name, const Type & type) const {
        const auto i = find(name, type);
        if (i == end()) {
            throw system_error(error_code(error::noSuchLabel),
                    "LabelSet::at: no {name:'" + name + "', type:'"
                    + type.title() + "'}");
        }
        return label(i);
    }

    template <typename ... Keys>
    bool exists(Keys&& ... keys) const {
        return find(std::forward<Keys>(keys)...) != end();
    }

    template <typename ... Keys>
    const Lid & lid(Keys&& ... keys) const {
        const auto i = find(std::forward<Keys>(keys)...);
        return i != end() ? label(i).lid() : null();
    }

    static const Lid & null() {
        static Lid retval;
        return retval;
    }
};

} // namespace macs
