#ifndef MACS_LABELS_REPOSITORY_H
#define MACS_LABELS_REPOSITORY_H

#include <map>
#include <list>

#include <macs/types.h>
#include <macs/label_factory.h>
#include <macs/hooks.h>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/thread.hpp>
#include <macs/user_journal.h>
#include <macs/label_set.h>
#include <macs/detail/cache.h>
#include <macs/io.h>

namespace macs {

/**
 * Labels storage abstraction
 */
class LabelsRepository : public std::enable_shared_from_this<LabelsRepository> {
public:

    LabelsRepository() = default;
    LabelsRepository(UserJournalPtr journal) : journal(journal) { }

    virtual ~LabelsRepository() = default;

    template <typename Handler = io::sync_context>
    auto getAllLabels(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnLabels> init{handler};
        labelsInternal([h = init.handler](error_code e, CachePtr c) mutable {
            if(e) {
                h(std::move(e), LabelSet());
            } else {
                h(std::move(e), *c);
            }
        });
        return init.result.get();
    }
    template <typename Handler = io::sync_context>
    auto createLabelWithType(const std::string& name,
                             const std::string& color,
                             const std::string& type,
                             Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateLabel> init{handler};
        createLabelWithTypeInternal(name, color, type, init.handler);
        return init.result.get();
    }

    auto createLabel(const std::string& name,
                      const std::string& color,
                      const std::string& type = "") const {
        return createLabelWithType(name, color, type);
    }

    template <typename Handler = io::sync_context>
    auto createLabel(const Label::Symbol & symbol,
                     Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateLabel> init{handler};
        createLabelInternal(symbol, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateLabel(Label label, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateLabel> init{handler};
        updateLabelInternal(std::move(label), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteLabel(const Lid& lid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init{handler};
        deleteLabelInternal(lid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto clearLabel(const Lid& lid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateMessages> init{handler};
        clearLabelInternal(lid, init.handler);
        return init.result.get();
    }

    void resetLabelsCache() const { cache.reset(); }

    template <typename Handler = io::sync_context>
    auto getThreadsCount(const Lid& lid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnCountReceive> init{handler};
        syncGetThreadsCount(lid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getOrCreateLabel(const std::string& name,
                          const std::string& color,
                          const Label::Type& type,
                          Handler handler = io::use_sync) const {

        io::detail::init_async_result<Handler, OnUpdateLabel> init{handler};
        getOrCreateLabelInternal(name, color, type, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getOrCreateLabel(const std::string& name,
                          const std::string& color,
                          Handler handler = io::use_sync) const {

        return getOrCreateLabel(name, color, Label::Type::getByTitle(""), handler);
    }

    template <typename Handler = io::sync_context>
    auto getOrCreateLabel(Label::Symbol symbol,
                          Handler handler = io::use_sync) const {

        io::detail::init_async_result<Handler, OnUpdateLabel> init{handler};
        getOrCreateLabelInternal(symbol, init.handler);
        return init.result.get();
    }

    // !!! Deprecated method getLabelByLid(), use LabelsSet::at() instead !!!
    Label getLabelByLid(const Lid& lid) const;
    // !!! Deprecated method getLabelByLids(), use LabelsSet::at() instead !!!
    template<typename Iterator, typename Inserter>
    Inserter getLabelsByLids(Iterator first, Iterator last, Inserter ins) const {
        const auto labels = labelsInternal();
        std::transform(first, last, ins, [&](auto& lid){ return labels->at(lid);});
        return ins;
    }
    // !!! Deprecated method existLabel(), use LabelsSet::exists() instead !!!
    bool existLabel(const Lid& lid) const;
    // !!! Deprecated method existLabel(), use LabelsSet::exists() instead !!!
    bool existLabel(const Label::Symbol & symbol) const;
    // !!! Deprecated method existLabelNameAndType(), use LabelsSet::exists() instead !!!
    bool existLabelNameAndType(const std::string & name, const Label::Type & type) const;
    // !!! Deprecated method getLabelLidBySymbol(), use FolderSet::lid() instead !!!
    std::string getLabelLidBySymbol(const Label::Symbol & symbol) const;
    // !!! Deprecated method getLabelBySymbol(), use FolderSet::at() instead !!!
    Label getLabelBySymbol(const Label::Symbol & symbol) const;
    // !!! Deprecated method getLabelLidByNameAndType(), use FolderSet::lid() instead !!!
    std::string getLabelLidByNameAndType(const std::string & name, const Label::Type & type) const;
    // !!! Deprecated method getLabelByNameAndType(), use FolderSet::at() instead !!!
    Label getLabelByNameAndType(const std::string& name, const Label::Type & type) const;

protected:
    virtual void syncGetLabels(OnLabels h) const = 0;

    virtual void syncCreateLabel(const std::string& name,
                                 const std::string& color,
                                 const Label::Type& type,
                                 OnUpdateLabel hook) const = 0;
    virtual void syncCreateLabel(const Label::Symbol & symbol, OnUpdateLabel hook) const = 0;
    virtual void syncModifyLabel(const Label& label, OnUpdateLabel hook) const = 0;

    virtual void syncEraseLabel(const std::string& lid, OnUpdate hook) const = 0;
    virtual void syncClearLabel(const Label& label, OnUpdateMessages hook) const = 0;

    virtual void syncGetThreadsCount(const std::string& lid, OnCountReceive hook) const = 0;

    virtual void syncGetOrCreateLabel(const std::string& name,
                                 const std::string& color,
                                 const Label::Type& type,
                                 OnUpdateLabel hook) const = 0;
    virtual void syncGetOrCreateLabel(const Label::Symbol & symbol, OnUpdateLabel hook) const = 0;

    LabelFactory getLabelfactory() const {
        return LabelFactory();
    }

private:
    void createLabelWithTypeInternal(const std::string& name,
                                 const std::string& color,
                                 const std::string& type,
                                 OnUpdateLabel hook) const;
    void createLabelInternal(const Label::Symbol& symbol, OnUpdateLabel hook) const;
    void updateLabelInternal(Label label, OnUpdateLabel hook) const;
    void deleteLabelInternal(const Lid& lid, OnUpdate hook) const;
    void clearLabelInternal(const Lid& lid, OnUpdateMessages hook) const;
    void getOrCreateLabelInternal(const std::string& name,
                                 const std::string& color,
                                 const Label::Type& type,
                                 OnUpdateLabel hook) const;
    void getOrCreateLabelInternal(const Label::Symbol& symbol, OnUpdateLabel hook) const;

    using LabelsCache = detail::Cache<LabelSet>;
    using CachePtr = LabelsCache::Ptr;

    CachePtr labelsInternal() const;

    template <typename Handler>
    void labelsInternal(Handler) const;

    template<typename OperationType, typename... ArgsT>
    void logOperation(ArgsT&& ... args) const {
        if( journal.get() ) {
            journal->logOperation<OperationType>(std::forward<ArgsT>(args)...);
        }
    }

    void logOnCreateLabel(const Label& label) const;

    mutable LabelsCache cache;
    UserJournalPtr journal;
};

template <typename Handler>
inline void LabelsRepository::labelsInternal(Handler handler) const {
    if(auto c = cache.get()) {
        handler(error_code(), c);
        return;
    }

    syncGetLabels([self = shared_from_this(), h = std::move(handler)]
                   (error_code e, LabelSet l) mutable {
        h(e, e ? self->cache.get() : self->cache.set(std::move(l)));
    });
}

typedef std::shared_ptr<LabelsRepository> LabelsRepositoryPtr;

} // namespace macs

#endif // MACS_LABELS_REPOSITORY_H
