/**
 * @file   labels_repository_imap.h
 * @brief  Labels repository for imap server.
 * @author egorp
 * © Yandex LLC.
 */
#ifndef MACS_LABELS_REPOSITORY_IMAP_H
#define MACS_LABELS_REPOSITORY_IMAP_H

#include <macs/labels_repository.h>
#include <macs/folders_repository.h>
#include <user_journal/parameters/imap.h>

namespace macs {


typedef Hook<ImapFolder> imap_folder_entry;

/** @class ImapLabelsRepository
 *  @brief labels repository for imap server
 *
 */
class ImapLabelsRepository : public std::enable_shared_from_this<ImapLabelsRepository> {
public:

    ImapLabelsRepository() { }
    ImapLabelsRepository(UserJournalPtr journal) : journal(journal) { }
    virtual ~ImapLabelsRepository() { }

    bool addImapFlag(const string& flagName);
    const LabelSet & getImapFlags();
    const FolderSet & getAllFolders();

    void getImapFolder(const std::string& fname, char delimiter, const imap_folder_entry& handler, bool timeSigned);
    void createImapFolder(const std::string& fname, char delimiter);
    void deleteImapFolder(const std::string& fname, char delimiter);
    void renameImapFolder(const std::string& oldName, const std::string& newName, char delimiter);

protected:
    virtual FolderSet asyncGetImapFolders() = 0;

    virtual void asyncGetImapFolder(const std::string& fname, char delimiter,
                                    const imap_folder_entry& handler,
                                    bool timeSigned) = 0;

    virtual void asyncCreateImapFolder(const std::string& fname, char delimiter) = 0;
    virtual void asyncDeleteImapFolder(const std::string& fname, char delimiter) = 0;
    virtual void asyncRenameImapFolder(const std::string& oldName,
                                       const std::string& newName,
                                       char delimiter) = 0;

    /// retrieve imap flags dictionary
    virtual LabelSet syncGetImapFlags() = 0;
    /// sync add new imap flag to dictionary
    virtual bool syncAddImapFlag(const string& flagName) = 0;

    LabelFactory getLabelfactory() { return LabelFactory(); }
    FolderFactory getFolderFactory() { return FolderFactory(); }

private:
    ImapLabelsRepository(const ImapLabelsRepository &);
    ImapLabelsRepository & operator=(const ImapLabelsRepository &);

    void fillFolders();
    void fillImapFlags();

    FolderSet& foldersInternal();

    template<typename OperationType, typename... ArgsT>
    void logOperation(ArgsT&& ... args) const {
        if( journal.get() ) {
            journal->logOperation<OperationType>(std::forward<ArgsT>(args)...);
        }
    }

    FolderSet folders;
    LabelSet imapFlags;
    UserJournalPtr journal;
};

inline const FolderSet & ImapLabelsRepository::getAllFolders() {
    return foldersInternal();
}

using boost::adaptors::filtered;

inline FolderSet& ImapLabelsRepository::foldersInternal() {
    if(folders.empty()) {
        fillFolders();
    }
    return folders;
}

inline void ImapLabelsRepository::getImapFolder(
    const std::string& fname, char delimiter,
    const imap_folder_entry& handler,
    bool timeSigned)
{
    asyncGetImapFolder(fname, delimiter, handler, timeSigned);
}

inline void ImapLabelsRepository::createImapFolder(const std::string& fname, char delimiter)
{
    asyncCreateImapFolder(fname, delimiter);
    using namespace user_journal::parameters;
    logOperation<ImapCreateFolder>(id::state(fname), id::affected(0ul));
}

inline void ImapLabelsRepository::deleteImapFolder(const std::string& fname, char delimiter)
{
    asyncDeleteImapFolder(fname, delimiter);
    using namespace user_journal::parameters;
    logOperation<ImapDeleteFolder>(id::state(fname), id::affected(0ul));
}

inline void ImapLabelsRepository::renameImapFolder(
    const std::string& oldName,
    const std::string& newName,
    char delimiter)
{
    asyncRenameImapFolder(oldName, newName, delimiter);
    using namespace user_journal::parameters;
    logOperation<ImapRenameFolder>(id::state(oldName + "->" + newName), id::affected(0ul));
}

inline const LabelSet & ImapLabelsRepository::getImapFlags()
{
    if ( imapFlags.empty() ) {
        fillImapFlags();
    }
    return imapFlags;
}

inline bool ImapLabelsRepository::addImapFlag(const string& flagName)
{
    return syncAddImapFlag(flagName);
}

inline void ImapLabelsRepository::fillFolders() {
    try {
        folders = asyncGetImapFolders();
    } catch(const std::exception& e) {
        throw system_error( error_code(macs::error::input,
                std::string("can't retrieve labels and folders: ") + e.what()) );
    }
}

inline void ImapLabelsRepository::fillImapFlags()
{
    try {
        imapFlags = syncGetImapFlags();
    } catch(const std::exception& e) {
        throw system_error( error_code(macs::error::input,
                std::string("can't retrieve imap flags: ") + e.what()) );
    }
}

typedef std::shared_ptr<ImapLabelsRepository> ImapLabelsRepositoryPtr;

}

#endif // MACS_LABELS_REPOSITORY_IMAP_H
