#ifndef MACS_ASYNC_CALLER_H
#define MACS_ASYNC_CALLER_H

#include <mail_errors/set_exception.h>
#include <macs/hooks.h>
#include <stdexcept>
#include <future>
#include <memory>
#include <iterator>

namespace macs {
namespace synchronize {
namespace detail {

template <typename Result>
inline auto callback(const std::string& name, std::promise<Result>& p) {
    return [&] (error_code err, Result res) {
        auto retval = std::move(p);
        if(err) {
            mail_errors::setSystemError(retval, name, err);
        } else {
            retval.set_value(std::move(res));
        }
    };
}

inline auto callback(const std::string& name, std::promise<void>& p) {
    return [&] (error_code err) {
        auto retval = std::move(p);
        if(err) {
            mail_errors::setSystemError(retval, name, err);
        } else {
            retval.set_value();
        }
    };
}

template <typename H>
struct argument;

template <typename T>
struct argument<macs::hooks::Hook<T>> {
    using type = typename macs::hooks::Hook<T>::second_argument_type;
};

template <>
struct argument<macs::hooks::Hook<>> {
    using type = void;
};

template <typename H>
using result = std::decay<typename argument<H>::type>;

template<class T, class Class, class ... MArgs, class ... Args>
auto exec(const Class* obj, const std::string& name,
        void (Class::*method)(MArgs...) const,
        Args&&... args) {
    std::promise<typename result<Hook<T>>::type> p;
    auto f = p.get_future();

    (obj->*method)(std::forward<Args>(args)..., callback(name, p));

    return f.get();
}

template<class C, class T>
auto execAsyncAndWait(const C* obj, const std::string& name,
                        void (C::*m)(Hook<T>) const) {
    return exec<T>(obj, name, m);
}

template<class C, class T, class A0, class P0>
auto execAsyncAndWait(const C* obj, const std::string& name,
                        void (C::*m)(A0, Hook<T>) const,
                        P0&& p0) {
    return exec<T>(obj, name, m, std::forward<P0>(p0));
}

template<class C, class T,
        class A0, class A1, class P0, class P1>
auto execAsyncAndWait(const C* obj, const std::string& name,
                        void (C::*m)(A0, A1, Hook<T>) const,
                        P0&& p0, P1&& p1){
    return exec<T>(obj, name, m, std::forward<P0>(p0), std::forward<P1>(p1));
}

template<class C, class T,
        class A0, class A1, class A2, class P0, class P1, class P2>
auto execAsyncAndWait(const C* obj, const std::string& name,
                        void (C::*m)(A0, A1, A2, Hook<T>) const,
                        P0&& p0, P1&& p1, P2&& p2){
    return exec<T>(obj, name, m, std::forward<P0>(p0), std::forward<P1>(p1),
            std::forward<P2>(p2));
}

template<class C, class T,
        class A0, class A1, class A2, class A3,
        class P0, class P1, class P2, class P3>
auto execAsyncAndWait(const C* obj, const std::string& name,
                        void (C::*m)(A0, A1, A2, A3, Hook<T>) const,
                        P0&& p0, P1&& p1, P2&& p2, P3&& p3){
    return exec<T>(obj, name, m, std::forward<P0>(p0), std::forward<P1>(p1),
            std::forward<P2>(p2), std::forward<P3>(p3));
}

} // namespace detail

template <typename Ins>
class Inserter {
public:
    Inserter(std::string name, Ins out) : name(std::move(name)), out(out) {}

    auto handler() {
        return [this] (error_code ec, auto item) {
            if (!ec && item) {
                *out++ = std::move(item.get());
            } else {
                auto res = std::move(promise);
                if(ec){
                    mail_errors::setSystemError(res, name, std::move(ec));
                } else {
                    res.set_value(out);
                }
            }
        };
    }

    auto get() { return future.get(); }

private:
    std::string name;
    Ins out;
    std::promise<Ins> promise;
    std::future<Ins> future = promise.get_future();
};


template <typename Ins>
inline auto makeInserter(std::string name, Ins out) {
    return Inserter<Ins>(std::move(name), out);
}
template <typename Container>
inline auto makeBackInserter(std::string name, Container& out) {
    return makeInserter(std::move(name), std::back_inserter(out));
}

} // namespace synchronize

using synchronize::detail::execAsyncAndWait;


} // namespace macs

#endif // MACS_ASYNC_CALLER_H
