#ifndef MACS_MIME_PART_FACTORY_H
#define MACS_MIME_PART_FACTORY_H

#include <macs/mime_part.h>
#include <boost/algorithm/string/case_conv.hpp>

namespace macs {

class MimePartFactory {
    using Data = MimePart::Data;
public:
#define MODIFIER(NAME) \
    MimePartFactory& NAME (decltype(Data::NAME) v) { \
        data().NAME = std::move(v); \
        return *this; \
    }
#define MODIFIER_WITH_LOWERCASE(NAME) \
    MimePartFactory& NAME (std::string v) { \
        boost::algorithm::to_lower(v); \
        data().NAME = std::move(v); \
        return *this; \
    }

    MODIFIER(hid)
    MODIFIER_WITH_LOWERCASE(contentType)
    MODIFIER_WITH_LOWERCASE(contentSubtype)
    MODIFIER(boundary)
    MODIFIER(name)
    MODIFIER(charset)
    MODIFIER(encoding)
    MODIFIER(contentDisposition)
    MODIFIER(fileName)
    MODIFIER(cid)
    MODIFIER(offsetBegin)
    MODIFIER(offsetEnd)

#undef MODIFIER
#undef MODIFIER_WITH_LOWERCASE

    MimePartFactory& reset() {
        data_.reset();
        return *this;
    }

    MimePart release() {
        checkValidity();
        auto retval = std::move(data_);
        reset();
        return MimePart{std::move(retval)};
    }

    MimePartFactory() = default;

    MimePartFactory(const MimePart& src)
        : data_(std::make_shared<Data>(src.data())) {
    }

    MimePartFactory(MimePart&& src)
        : data_(std::const_pointer_cast<Data>(src.data_)) {
    }

    MimePartFactory(const MimePartFactory&) = delete;
    MimePartFactory(MimePartFactory&&) = default;
    MimePartFactory& operator=(const MimePartFactory&) = delete;
    MimePartFactory& operator=(MimePartFactory&&) = default;

private:
    std::shared_ptr<Data> data_;

    Data& data() {
        if (!data_) {
            data_ = std::make_shared<Data>();
        }
        return *data_;
    }

    const Data& data() const {
        return data_ ? *data_ : Data::default_;
    }

    void checkValidity() const {
        if (data().offsetBegin > data().offsetEnd) {
            throw std::runtime_error("Bad MimePart: offsetEnd is less than offsetBegin");
        }
    }
};

}

#endif // MACS_MIME_PART_FACTORY_H
