#pragma once

#include <list>
#include <string>

#include <macs/label.h>
#include <macs/folder.h>
#include <macs/queries/envelopes_query.h>

namespace macs {

class EnvelopesRepository;

class EnvelopesQueryWithMimesInFolder {
public:
    EnvelopesQueryWithMimesInFolder(const EnvelopesRepository& repo, const Fid& fid)
        : repository_(repo), fid_(fid) {
    }
    EnvelopesQueryWithMimesInFolder& from(const size_t row) {
        from_ = row;
        return *this;
    }
    EnvelopesQueryWithMimesInFolder& count(const size_t count) {
        count_ = count;
        return *this;
    }
    EnvelopesQueryWithMimesInFolder& sortBy(const EnvelopesSorting & order) {
        order_ = order;
        return *this;
    }

    template <typename Handler>
    auto get(Handler handler) const {
        io::detail::init_async_result<Handler, OnEnvelopeWithMimeReceive> init(handler);
        byEntry(init.handler);
        return init.result.get();
    }
private:
    const EnvelopesRepository& repository_;
    void byEntry (OnEnvelopeWithMimeReceive) const;
    Fid fid_;
    size_t from_ = 0;
    size_t count_ = std::numeric_limits<std::size_t>::max();
    EnvelopesSorting order_;
};

class EnvelopesQueryWithMimesInFolderByImapId {
public:
    using Self = EnvelopesQueryWithMimesInFolderByImapId;

    EnvelopesQueryWithMimesInFolderByImapId(const EnvelopesRepository& repo, const Fid& fid)
        : repository_(repo), fid_(fid) {
    }
    Self& from(const int64_t imapId) {
        minImapId_ = imapId;
        return *this;
    }
    Self& to(const int64_t imapId) {
        maxImapId_ = imapId;
        return *this;
    }
    Self& count(const std::size_t count) {
        count_ = count;
        return *this;
    }

    template <typename Handler>
    auto get(Handler handler) const {
        io::detail::init_async_result<Handler, OnEnvelopeWithMimeChunkReceive> init(handler);
        byEntry(init.handler);
        return init.result.get();
    }
private:
    void byEntry(OnEnvelopeWithMimeChunkReceive) const;

    const EnvelopesRepository& repository_;
    Fid fid_;
    int64_t minImapId_ = 0;
    int64_t maxImapId_ = std::numeric_limits<int64_t>::max();
    std::size_t count_ = std::numeric_limits<std::size_t>::max();
};

class EnvelopesQueryWithMimesByMids {
public:
    EnvelopesQueryWithMimesByMids(const EnvelopesRepository& repo, const MidList& mids)
        : repository_(repo), mids_(mids) {
    }

    template <typename Handler>
    auto get(Handler handler) const {
        io::detail::init_async_result<Handler, OnEnvelopeWithMimeReceive> init(handler);
        byEntry(init.handler);
        return init.result.get();
    }
private:
    const EnvelopesRepository& repository_;
    void byEntry (OnEnvelopeWithMimeReceive) const;
    MidList mids_;
};

class EnvelopesQueryWithMimes {
public:
    EnvelopesQueryWithMimes(const EnvelopesRepository& repo)
        : repository_(repo) {
    }
    EnvelopesQueryWithMimesInFolder inFolder(const Fid& fid) const {
        return EnvelopesQueryWithMimesInFolder(repository_, fid);
    }
    EnvelopesQueryWithMimesInFolderByImapId byImapId(const Fid& fid) const {
        return EnvelopesQueryWithMimesInFolderByImapId(repository_, fid);
    }
    EnvelopesQueryWithMimesByMids byMids(const MidList& mids) const {
        return EnvelopesQueryWithMimesByMids(repository_, mids);
    }
private:
    const EnvelopesRepository& repository_;
};

} // macs

