#pragma once

#include <optional>
#include <vector>
#include <map>
#include <memory>
#include <string>

namespace macs::settings {

enum class Mode {
    production,
    corp
};

using SettingsList = std::vector<std::string>;
using SettingsMap = std::map<std::string, std::string>;

struct TextTraits {
    TextTraits() = default;

    TextTraits(std::int32_t code_, std::int32_t lang_)
        : code(code_)
        , lang(lang_) {}

    std::int32_t code = -1;
    std::int32_t lang = 0;
};

struct Signature {
    std::optional<std::string> text = std::string();
    std::optional<std::string> lang;
    std::vector<std::string> associated_emails;
    bool is_default = false;
    std::optional<TextTraits> text_traits = TextTraits();
};

using SignaturesList = std::vector<Signature>;
using SignaturesListOpt = std::optional<SignaturesList>;

struct Profile {
    SignaturesListOpt signs;
    SettingsMap single_settings;

    Profile(const SignaturesListOpt& signs_ , const SettingsMap& single_settings_)
            : signs(signs_)
            , single_settings(single_settings_) {}

    Profile(SignaturesListOpt&& signs_ , SettingsMap&& single_settings_)
            : signs(std::move(signs_))
            , single_settings(std::move(single_settings_)) {}

    Profile() = default;
};

struct Parameters {
    SettingsMap single_settings;

    Parameters(const SettingsMap& single_settings_)
            : single_settings(single_settings_) {}

    Parameters(SettingsMap&& single_settings_)
            : single_settings(std::move(single_settings_)) {}

    Parameters() = default;
};

struct Settings {
    Profile profile;
    Parameters parameters;

    Settings(Profile&& profile_ , Parameters&& parameters_)
            : profile(std::move(profile_))
            , parameters(std::move(parameters_)) {}

    Settings() = default;
};

using ProfilePtr = std::shared_ptr<Profile>;
using ParametersPtr = std::shared_ptr<Parameters>;
using SettingsPtr = std::shared_ptr<Settings>;

}
