#pragma once

#include <macs/settings.h>

#include <unordered_map>
#include <set>
#include <optional>
#include <functional>

namespace macs::settings {

class Description {
public:
    enum class Type {
        flag,
        chr,
        no_type
    };

    struct Limits {
        std::int32_t min;
        std::int32_t max;
    };

    enum class Updatability {
        readonly,
        defended,
        updated
    };

    using Values = std::set<std::string>;

    struct Restriction {
        std::optional<std::string> onInit;
        Values values;
        std::optional<Limits> limits;
        Updatability updatability;
        Type type;
    };

    using Restrictions = std::unordered_map<std::string, Restriction>;

    struct SettingsRestrictions {
        Restrictions profile;
        Restrictions parameters;
    };

    Description();
    SettingsMap getProfile(const SettingsMap&) const;
    SettingsMap getParameters(const SettingsMap&) const;
    bool isParameters(const SettingsMap&) const;
    bool isParameters(const SettingsList&) const;
    bool isProtectedParameters(const SettingsMap&) const;
    SettingsMap validateProfile(const SettingsMap&) const;
    SettingsMap::value_type validateSetting(
            const std::pair<Restrictions::const_iterator, SettingsMap::value_type>&) const;

    SettingsMap::mapped_type setFlag(const SettingsMap::mapped_type& v) const {
        return v == "on" ? "on" : "";
    }

    const SettingsMap& initProfile() const {
        switch (mode) {
            case Mode::production:
                return initProductionProfile;
            case Mode::corp:
                return initCorpProfile;
            default:
                return initProductionProfile;
        }
    }

    const SettingsMap& initParameters() const {
        switch (mode) {
            case Mode::production:
                return initProductionParameters;
            case Mode::corp:
                return initCorpParameters;
            default:
                return initProductionParameters;
        }
    }

    const SettingsMap& initSettings() const {
        switch (mode) {
            case Mode::production:
                return initProductionSettings;
            case Mode::corp:
                return initCorpSettings;
            default:
                return initProductionSettings;
        }
    }

    const SettingsRestrictions& getRestrictions() const {
        switch (mode) {
            case Mode::production:
                return production;
            case Mode::corp:
                return corp;
            default:
                return production;
        }
    }

    void setMode(Mode mode) {
        this->mode = mode;
    }

private:
    SettingsMap getInitValues(const Restrictions&);

    SettingsMap::value_type validateFlag(
        const SettingsMap::value_type&,
        const Restrictions::const_iterator&,
        const std::string&,
        Type
    ) const;

    SettingsMap::value_type validateLimits(
        const SettingsMap::value_type&,
        const Limits&,
        const std::string&
    ) const;

    SettingsMap::value_type validateValues(
        const SettingsMap::value_type&,
        const Values&,
        const std::string&
    ) const;

    SettingsMap initProductionProfile;
    SettingsMap initProductionParameters;
    SettingsMap initProductionSettings;
    SettingsMap initCorpProfile;
    SettingsMap initCorpParameters;
    SettingsMap initCorpSettings;

    static const SettingsRestrictions production;
    static const SettingsRestrictions corp;
    Mode mode = Mode::production;
};

}
