#pragma once

#include <macs/error_code.h>
#include <macs/errors.h>
#include <macs/hooks.h>
#include <macs/io.h>

#include <macs/settings.h>
#include <macs/settings_description.h>
#include <macs/settings_utils.h>

namespace macs {

using namespace macs::settings;

class SettingsRepository : public std::enable_shared_from_this<SettingsRepository> {
public:

    virtual ~SettingsRepository() = default;

    template <typename Handler = io::sync_context>
    auto getSettings(SettingsList requestedSettings, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnSettings> init(handler);
        asyncGetSettings(
            [self = shared_from_this(), hook = init.handler, requestedSettings = std::move(requestedSettings)]
                    (sys::error_code ec, auto settings) mutable {
                if (ec) {
                    hook(std::move(ec), {});
                    return;
                }
                if (!settings) {
                    hook(error_code(error::noSettings, "settings not received"), {});
                    return;
                }
                Profile profile;
                Parameters parameters;

                utils::addSignatures(profile, settings->signs, requestedSettings);
                if (settings->single_settings) {
                    auto& singleSettings = *settings->single_settings;
                    utils::filterSettings(singleSettings, requestedSettings);
                    profile.single_settings = self->description.getProfile(singleSettings);
                    parameters.single_settings = self->description.getParameters(singleSettings);
                } else {
                    profile.single_settings = self->description.initProfile();
                    parameters.single_settings = self->description.initParameters();
                    utils::filterSettings(profile.single_settings, requestedSettings);
                    utils::filterSettings(parameters.single_settings, requestedSettings);
                }
                hook(error_code(), std::make_shared<Settings>(std::move(profile), std::move(parameters)));
            }
        );
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getProfile(SettingsList requestedSettings, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnProfile> init(handler);
        asyncGetSettings(
            [self = shared_from_this(), hook = init.handler, requestedSettings = std::move(requestedSettings)]
                    (sys::error_code ec, auto settings) mutable {
                if (ec) {
                    hook(std::move(ec), {});
                    return;
                }
                if (!settings) {
                    hook(error_code(error::noSettings, "settings not received"), {});
                    return;
                }
                Profile profile;

                utils::addSignatures(profile, settings->signs, requestedSettings);
                if (settings->single_settings) {
                    auto& singleSettings = *settings->single_settings;
                    utils::filterSettings(singleSettings, requestedSettings);
                    profile.single_settings = self->description.getProfile(singleSettings);
                } else {
                    profile.single_settings = self->description.initProfile();
                    utils::filterSettings(profile.single_settings, requestedSettings);
                }
                hook(error_code(), std::make_shared<Profile>(std::move(profile)));
            }
        );
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getParameters(SettingsList requestedSettings, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnParameters> init(handler);
        asyncGetSettings(
            [self = shared_from_this(), hook = init.handler, requestedSettings = std::move(requestedSettings)]
                    (sys::error_code ec, auto settings) mutable {
                if (ec) {
                    hook(std::move(ec), {});
                    return;
                }
                if (!settings) {
                    hook(error_code(error::noSettings, "settings not received"), {});
                    return;
                }
                Parameters parameters;

                if (settings->single_settings) {
                    auto& singleSettings = *settings->single_settings;
                    utils::filterSettings(singleSettings, requestedSettings);
                    parameters.single_settings = self->description.getParameters(singleSettings);
                } else {
                    parameters.single_settings = self->description.initParameters();
                    utils::filterSettings(parameters.single_settings, requestedSettings);
                }
                hook(error_code(), std::make_shared<Parameters>(std::move(parameters)));
            }
        );
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto initSettings(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<bool>> init(handler);
        asyncInitSettings(
            SettingsRaw {description.initSettings()},
            [hook = init.handler] (sys::error_code ec, auto count) mutable {
                if (ec) {
                    hook(std::move(ec), {});
                    return;
                }
                count == 0 ? hook(error_code(), false)
                    : hook(error_code(), true);
            }
        );
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateParameters(ParametersPtr parameters, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<bool>> init(handler);
        if (!description.isParameters(parameters->single_settings) || description.isProtectedParameters(parameters->single_settings)) {
            init.handler(error_code(error::invalidArgument, "invalid argument"), false);
        } else {
            asyncUpdateSettings(
                SettingsRaw {parameters->single_settings},
                [hook = init.handler] (sys::error_code ec, auto count) mutable {
                    if (ec) {
                        hook(std::move(ec), {});
                        return;
                    }
                    count == 0 ? hook(error_code(), false)
                        : hook(error_code(), true);
                }
            );
        }
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateProtectedParameters(ParametersPtr parameters, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<bool>> init(handler);
        if (!description.isParameters(parameters->single_settings)) {
            init.handler(error_code(error::invalidArgument, "invalid argument"), false);
        } else {
            asyncUpdateSettings(
                SettingsRaw {parameters->single_settings},
                [hook = init.handler] (sys::error_code ec, auto count) mutable {
                    if (ec) {
                        hook(std::move(ec), {});
                        return;
                    }
                    count == 0 ? hook(error_code(), false)
                        : hook(error_code(), true);
                }
            );
        }
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateProfile(ProfilePtr profile, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<bool>> init(handler);
        profile->single_settings = description.validateProfile(profile->single_settings);
        asyncUpdateSettings(
            SettingsRaw {
                profile->single_settings,
                profile->signs
            },
            [hook = init.handler] (sys::error_code ec, auto count) mutable {
                if (ec) {
                    hook(std::move(ec), {});
                    return;
                }
                count == 0 ? hook(error_code(), false)
                    : hook(error_code(), true);
            }
        );
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto deleteSettings(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<bool>> init(handler);
        asyncDeleteSettings(
            [hook = init.handler] (sys::error_code ec, auto count) mutable {
                if (ec) {
                    hook(std::move(ec), {});
                    return;
                }
                count == 0 ? hook(error_code(), false)
                    : hook(error_code(), true);
            }
        );
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto eraseParameters(SettingsList deletedParameters, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<bool>> init(handler);
        if (!description.isParameters(deletedParameters)) {
            init.handler(error_code(error::invalidArgument, "invalid argument"), false);
        } else {
            asyncEraseSettings(
                std::move(deletedParameters),
                [hook = init.handler] (sys::error_code ec, auto count) mutable {
                    if (ec) {
                        hook(std::move(ec), {});
                        return;
                    }
                    count == 0 ? hook(error_code(), false)
                        : hook(error_code(), true);
                }
            );
        }
        return init.result.get();
    }

    std::shared_ptr<SettingsRepository> setMode(Mode mode) {
        description.setMode(mode);
        return shared_from_this();
    }

protected:
    virtual void asyncGetSettings(OnQuerySettings) const = 0;
    virtual void asyncInitSettings(SettingsRaw, Hook<std::int32_t>) const = 0;
    virtual void asyncUpdateSettings(SettingsRaw, Hook<std::int32_t>) const = 0;
    virtual void asyncDeleteSettings(Hook<std::int32_t>) const = 0;
    virtual void asyncEraseSettings(SettingsList, Hook<std::int32_t>) const = 0;

    Description description;
};

using  SettingsRepositoryPtr = std::shared_ptr<SettingsRepository>;

}
