#pragma once

#include <macs/hooks.h>
#include <macs/io.h>
#include <macs/types.h>
#include <macs/user_journal.h>

namespace macs {

class StickersRepository : public std::enable_shared_from_this<StickersRepository> {
public:
    StickersRepository() = default;
    StickersRepository(UserJournalPtr journal) : journal(std::move(journal)) { }

    virtual ~StickersRepository() = default;

    template <typename Handler = io::sync_context>
    auto getReplyLaterList(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnReplyLaterStickers> init(handler);
        asyncGetReplyLaterList(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto createReplyLater(const Mid& mid, const Fid& fid, std::time_t date, const StickerAllowedInterval& allowedInterval, const std::optional<Tab::Type>& tab, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnReplyLaterSticker> init(handler);
        createReplyLaterInternal(mid, fid, date, allowedInterval, tab, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto removeReplyLater(const Mid& mid, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        removeReplyLaterInternal(mid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto removeIncorrectReplyLaterStickers(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnIncorrectStickersMids> init(handler);
        asyncRemoveIncorrectReplyLaterStickers(init.handler);
        return init.result.get();
    }

    static void checkDateInInterval(std::time_t date, std::time_t now, const StickerAllowedInterval& allowedInterval);

protected:
    virtual void asyncGetReplyLaterList(OnReplyLaterStickers) const = 0;
    virtual void asyncCreateReplyLater(const Mid&, const Fid&, std::time_t date, const std::optional<Tab::Type>&, OnReplyLaterSticker) const = 0;
    virtual void asyncRemoveReplyLater(const Mid& mid, OnUpdate) const = 0;
    virtual void asyncRemoveIncorrectReplyLaterStickers(OnIncorrectStickersMids) const = 0;

private:
    void createReplyLaterInternal(const Mid& mid, const Fid& fid, std::time_t date, const StickerAllowedInterval& allowedInterval, const std::optional<Tab::Type>&, OnReplyLaterSticker hook) const;
    void removeReplyLaterInternal(const Mid& mid, OnUpdate hook) const;

    void logOnRemoveSticker(const Mid& mid, macs::StickerType stickerType) const;

    template<typename OperationType, typename... ArgsT>
    void logOperation(ArgsT&& ... args) const {
        if(journal) {
            journal->logOperation<OperationType>(std::forward<ArgsT>(args)...);
        }
    }

    UserJournalPtr journal;
};

using StickersRepositoryPtr = std::shared_ptr<StickersRepository>;

}
