#pragma once

#include <string>
#include <macs/revision.h>
#include <macs/errors.h>
#include <user_journal/enumeration.h>
#include <boost/assign.hpp>

namespace macs {

struct TabData {
    struct __TabType{
        enum Enum {
            unknown,
            relevant,
            news,
            social
        };

        using Map = user_journal::Enum2String<Enum>::Map;

        void fill(Map & map) const{
            boost::assign::insert(map)
                ( relevant, "relevant" )
                ( news, "news" )
                ( social, "social" );
        }

        using Filler = __TabType;
    };

    using Type = user_journal::Enumeration<__TabType>;

    Type type;

    size_t bytes = 0;
    size_t messagesCount = 0;
    size_t newMessagesCount = 0;
    size_t freshMessagesCount = 0;

    // additional
    std::time_t creationTime;
    Revision revision = NULL_REVISION;

    static const TabData default_;
};

template <typename Impl>
class TabDataInterface {
public:
#define GETTER(NAME) auto& NAME() const { return data().NAME; }
    GETTER(type)
    GETTER(bytes)
    GETTER(messagesCount)
    GETTER(newMessagesCount)
    GETTER(freshMessagesCount)
    GETTER(creationTime)
    GETTER(revision)
#undef GETTER
private:
    friend Impl;
    TabDataInterface() = default;
    TabDataInterface(const TabDataInterface&) = default;
    TabDataInterface(TabDataInterface&&) = default;
    TabDataInterface& operator = (const TabDataInterface&) = default;
    TabDataInterface& operator = (TabDataInterface&&) = default;

    const TabData& data() const {
        return static_cast<const Impl&>(*this).Impl::data();
    }
};

class Tab: public TabDataInterface<Tab> {
public:
    Tab() = default;
    Tab(std::shared_ptr<const TabData> data): data_(std::move(data))
        {}
    Tab(const Tab&) = default;
    Tab(Tab&&) = default;
    Tab& operator=(const Tab&) = default;
    Tab& operator=(Tab&&) = default;

    using Type = TabData::Type;

protected:
    friend class TabFactory;
    friend class TabDataInterface<Tab>;
    const TabData& data() const {
        return data_ ? *data_ : TabData::default_;
    }
    std::shared_ptr<const TabData> data_;
};

} // namespace macs
