#pragma once

#include <macs/tab.h>
#include <macs/tab_factory.h>
#include <map>

namespace macs {

using TabsMap = std::map<Tab::Type,Tab>;

class TabSet {
public:
    using value_type = TabsMap::value_type;
    using size_type = TabsMap::size_type;
    using const_iterator = TabsMap::const_iterator;
    using iterator = const_iterator;
    using mapped_type = TabsMap::mapped_type;

    TabSet() = default;

    TabSet(TabsMap data)
        : TabSet(std::make_shared<TabsMap>(std::move(data))) {
    }

    TabSet(std::shared_ptr<const TabsMap> data)
        : data(std::move(data)) {
    }

    const_iterator begin() const {
        return data ? data->begin() : emptyMap.cend();
    }

    const_iterator end() const {
        return data ? data->end() : emptyMap.cend();
    }

    bool empty() const {
        return data ? data->empty() : true;
    }

    size_type size() const {
        return data ? data->size() : 0;
    }

    const_iterator find(const Tab::Type& type) const {
        return data ? data->find(type) : emptyMap.cend();
    }

    const_iterator find(const std::string& type) const {
        return find(Tab::Type::fromString(type, std::nothrow));
    }

    template <typename Key>
    bool exists(Key&& key) const {
        return find(std::forward<Key>(key)) != end();
    }

    template <typename Key>
    const Tab& at(const Key& key) const {
        const auto it = find(key);
        if (it == end()) {
            throw system_error(error_code(error::noSuchTab),
                    "access to nonexistent tab '"
                    + printKey(key) + "'");
        }
        return it->second;
    }

private:
    std::string printKey(const Tab::Type& type) const {
        return type.toString();
    }

    std::string printKey(const std::string& type) const {
        return type;
    }

    std::shared_ptr<const TabsMap> data;

    static const TabsMap emptyMap;
};

} // namespace macs
