#pragma once

#include <map>

#include <macs/types.h>
#include <macs/tab_factory.h>
#include <macs/hooks.h>
#include <boost/scoped_ptr.hpp>
#include <boost/thread.hpp>
#include <macs/user_journal.h>
#include <macs/tab_set.h>
#include <macs/detail/cache.h>
#include <boost/optional.hpp>
#include <macs/io.h>

namespace macs {

/// Tabs access and manipulation class
class TabsRepository : public std::enable_shared_from_this<TabsRepository> {
public:
    TabsRepository() = default;

    virtual ~TabsRepository() = default;

    void resetTabsCache() const {cache.reset();}

    template <typename Handler = io::sync_context>
    auto getAllTabs(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnTabs> init(handler);
        tabsInternal([h = init.handler](error_code e, CachePtr c) mutable {
            if(e) {
                h(std::move(e), TabSet{});
            } else {
                h(error_code(), *c);
            }
        });
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getOrCreateTab(const Tab::Type& type,
                        Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdateTab> init(handler);
        getOrCreateTabInternal(type, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto resetFresh(const Tab& tab,
                    Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnUpdate> init(handler);
        resetFreshInternal(tab, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto canReadTabs(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, Hook<bool>> init(handler);
        asyncCanReadTabs(init.handler);
        return init.result.get();
    }

protected:
    void getOrCreateTabInternal(const Tab::Type& type, OnUpdateTab hook) const;
    void resetFreshInternal(const Tab& tab, OnUpdate hook) const;

    virtual void asyncGetTabs(OnTabs hook) const = 0;
    virtual void asyncGetOrCreateTab(const Tab::Type& type, OnUpdateTab hook) const = 0;
    virtual void asyncResetFresh(const Tab::Type& tab, OnUpdate hook) const = 0;
    virtual void asyncCanReadTabs(Hook<bool> hook) const = 0;

    TabFactory getTabFactory() const {
        return TabFactory();
    }

private:
    using TabsCache = detail::Cache<TabSet>;
    using CachePtr = TabsCache::Ptr;

    CachePtr tabsInternal() const;
    template <typename Handler>
    void tabsInternal(Handler handler) const {
        if(auto c = cache.get()) {
            handler(error_code(), c);
            return;
        }

        asyncGetTabs([self = shared_from_this(), h = std::move(handler)]
                       (error_code e, TabSet v) mutable {
            if (e) {
                h(std::move(e), self->cache.get());
            } else {
                h(std::move(e), self->cache.set(std::move(v)));
            }
        });
    }

    mutable TabsCache cache;
};

typedef std::shared_ptr<TabsRepository> TabsRepositoryPtr;

} // namespace macs
