#pragma once

#include <macs/thread_participants_factory.h>
#include <macs/thread_labels.h>
#include <macs/hooks.h>
#include <macs/thread_mailbox_list.h>
#include <macs/method_caller.h>
#include <macs/types.h>
#include <future>
#include <macs/threads_meta.h>
#include <macs/io.h>
#include <macs/check_arguments.h>

namespace macs {

/// Threads access class
class ThreadsRepository : public std::enable_shared_from_this<ThreadsRepository> {
public:

    virtual ~ThreadsRepository() = default;

    /// get threads for given list of tids
    template <typename Handler = io::sync_context>
    auto getThreadsParticipants(const TidVector& tids, Handler handler = io::use_sync) const {
        ASSERT_NOT_EMPTY_ARG(tids);
        ASSERT_BIGINT_ITEMS(tids);
        io::detail::init_async_result<Handler, OnParticipantsReceive> init(handler);
        syncThreadsParticipantsList(tids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getThreadLabels(const TidVector& tids, Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(tids);
        io::detail::init_async_result<Handler, OnThreadLabelsReceive> init(handler);
        syncThreadLabelsList(tids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto fillIdsMap(const MidList& mids, const TidList& tids, const Lids& lidsToCheck,
            Handler handler = io::use_sync) const{
        ASSERT_BIGINT_ITEMS(mids);
        ASSERT_BIGINT_ITEMS(tids);

        io::detail::init_async_result<Handler, OnThreadMailboxItemsReceive> init(handler);
        syncFillIdsMap(mids, tids, lidsToCheck, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto fillIdsMap(const MidList& mids, const TidList& tids, Handler handler = io::use_sync) const {
        return fillIdsMap(mids, tids, {}, handler);
    }

    template <typename Handler = io::sync_context>
    auto fillIdsList(const MidList& mids, const TidList& tids,
            Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(mids);
        ASSERT_BIGINT_ITEMS(tids);

        io::detail::init_async_result<Handler, OnMidsReceive> init(handler);
        syncFillIdsList(mids, tids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto fillIdsListWithoutStatus(const MidList& mids, const TidList& tids,
            macs::Envelope::Status exclude, Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(mids);
        ASSERT_BIGINT_ITEMS(tids);

        io::detail::init_async_result<Handler, OnMidsReceive> init(handler);
        syncFillIdsListWithoutStatus(mids, tids, exclude, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto fillIdsFList(const TidList& tidsf, const Fid& fid,
            Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(tidsf);
        ASSERT_INTEGER_ARG(fid);

        io::detail::init_async_result<Handler, OnMidsReceive> init(handler);
        syncFillIdsFList(tidsf, fid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto findThreadsByHash(const ThreadHash& hash, const ThreadLimits& limits,
            Handler handler = io::use_sync) const {

        io::detail::init_async_result<Handler, OnThreadIdsReceive> init(handler);
        syncFindThreadsByHash(hash, limits, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto findThreadsByReferences(const MidVec& mids, const MailRefVec& refs,
            Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(mids);

        io::detail::init_async_result<Handler, OnThreadIdsReceive> init(handler);
        syncFindThreadsByReferences(mids, refs, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto joinThreads( const Tid & tid, const TidVector& joinTids,
            Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ARG(tid);
        ASSERT_BIGINT_ITEMS(joinTids);

        io::detail::init_async_result<Handler, OnThreadIdsReceive> init(handler);
        syncJoinThreads(tid, joinTids, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto filterThreadsByLid (const TidVector& tids, const Lid& lid,
            Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(tids);
        ASSERT_INTEGER_ARG(lid);

        io::detail::init_async_result<Handler, OnThreadIdsReceive> init(handler);
        syncFilterThreadsByLid(tids, lid, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getMessageCountInThreads(const TidVector& tids, Handler handler = io::use_sync) const {
        ASSERT_BIGINT_ITEMS(tids);

        io::detail::init_async_result<Handler, OnCountReceive> init(handler);
        syncMessageCountInThreads(tids, init.handler);
        return init.result.get();
    }
protected:
    /// get threads for given list of tids
    virtual void syncThreadsParticipantsList(const TidVector& tid,
                                             OnParticipantsReceive handler) const = 0;

    virtual void syncThreadLabelsList(const TidVector& tids,
                                      OnThreadLabelsReceive handler) const = 0;

    ThreadParticipantsFactory getThreadParticipantsFactory() const {
        return ThreadParticipantsFactory ();
    }

    virtual void syncFillIdsMap(const MidList& mids, const TidList& tids, const Lids& lidsToCheck,
                                OnThreadMailboxItemsReceive handler) const = 0;

    virtual void syncFillIdsList(const MidList& mids,
                                 const TidList& tids,
                                 OnMidsReceive handler) const = 0;

    virtual void syncFillIdsListWithoutStatus(const MidList& mids,
                                              const TidList& tids,
                                              macs::Envelope::Status excludeStatus,
                                              OnMidsReceive handler) const = 0;

    virtual void syncFillIdsFList(const TidList& tidsf,
                                  const Fid& fid,
                                  OnMidsReceive handler) const = 0;

    virtual void syncFindThreadsByHash(const ThreadHash & hash,
            const ThreadLimits & limits,
            OnThreadIdsReceive handler) const = 0;

    virtual void syncFindThreadsByReferences(const MidVec & mids,
            const MailRefVec & refs,
            OnThreadIdsReceive handler) const = 0;

    virtual void syncJoinThreads( const ThreadId & tid, const TidVector& joinTids,
            OnThreadIdReceive handler) const = 0;

    virtual void syncFilterThreadsByLid (const TidVector& tids, const Lid& lid,
            OnThreadIdsReceive handler) const = 0;

    virtual void syncMessageCountInThreads(const TidVector& tids,
            OnCountReceive handler) const = 0;
};

typedef std::shared_ptr<ThreadsRepository> ThreadsRepositoryPtr;

} // namespace macs
