#pragma once

#include <macs/hooks.h>
#include <macs/io.h>
#include <macs/types.h>

namespace macs {

/// Provides info related to the user itself (like the `state` column of the `mail.users` table)
/// not to his messages, subscriptions etc.
class UsersRepository {
public:
    virtual ~UsersRepository() = default;

    template <typename Handler = io::sync_context>
    auto getFreezingInfo(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnFreezingInfo> init(handler);
        asyncFreezingInfo(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto unfreezeUser(Handler handler = io::use_sync) const{
        io::detail::init_async_result<Handler, OnExecute> init{handler};
        asyncChangeUserState(UserState::frozen, UserState::active, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto getArchive(Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, onArchive> init(handler);
        asyncGetArchive(init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto activateUserAndUpdateArchiveState(ArchiveState from, ArchiveState to, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncActivateUserAndUpdateArchiveState(from, to, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto exchangeArchiveTaskId(TaskId from, TaskId to, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncExchangeArchiveTaskId(from, to, init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto exchangeArchiveState(UserState userState, ArchiveState from, ArchiveState to, 
                std::optional<std::string> notice = {}, Handler handler = io::use_sync) const {

        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncExchangeArchiveState(userState, from, to, std::move(notice), init.handler);
        return init.result.get();
    }

    template <typename Handler = io::sync_context>
    auto updateRestorationProgress(TaskId fromTaskId, std::int32_t restoredMessageCount, Handler handler = io::use_sync) const {

        io::detail::init_async_result<Handler, OnExecute> init(handler);
        asyncUpdateRestorationProgress(fromTaskId, restoredMessageCount, init.handler);
        return init.result.get();
    }


protected:
    virtual void asyncFreezingInfo(OnFreezingInfo handler) const = 0;
    virtual void asyncChangeUserState(UserState fromState, UserState toState, OnExecute handler) const = 0;
    virtual void asyncGetArchive(onArchive handler) const = 0;
    virtual void asyncActivateUserAndUpdateArchiveState(ArchiveState fromState, ArchiveState toState, OnExecute handler) const = 0;
    virtual void asyncExchangeArchiveTaskId(TaskId from, TaskId to, OnExecute handler) const = 0;
    virtual void asyncExchangeArchiveState(UserState userState, ArchiveState from, ArchiveState to,
                                        std::optional<std::string> notice, OnExecute handler) const = 0;

    virtual void asyncUpdateRestorationProgress(TaskId fromTaskId, std::int32_t restoredMessageCount, OnExecute handler) const = 0;
};

using UsersRepositoryPtr = std::shared_ptr<UsersRepository>;

}
