#include <string>
#include <list>
#include <future>
#include <boost/system/system_error.hpp>
#include <boost/bind.hpp>
#include <boost/algorithm/string/join.hpp>
#include <boost/range/adaptors.hpp>
#include <boost/bind.hpp>
#include <user_journal/parameters/mailbox.h>
#include <user_journal/parameters/message.h>

#include <macs/types.h>
#include <macs/envelopes_repository.h>
#include <macs/queries/envelopes_query.h>

#include <future>

using std::string;

namespace macs {

using Oper = UserJournal::Operation;
namespace params = user_journal::parameters;
namespace id = params::id;

void EnvelopesRepository::getByIdInternal(const Mid& id, Hook<Envelope> handler) const {
    ASSERT_BIGINT_ARG(id);
    auto result = std::make_shared<boost::optional<Envelope>>();
    syncEntriesByIds({id}, [=] (error_code ec, auto cursor) {
        if (!cursor) {
            if (result->is_initialized()) {
                handler(std::move(result->get()));
            } else if(!ec) {
                handler(error_code(error::noSuchMessage, "no such message " + id));
            } else {
                handler(std::move(ec));
            }
        } else if (!result->is_initialized()) {
            *result = std::move(*cursor);
        }
    });
}

void EnvelopesRepository::saveInternal( Envelope envelope, MimeParts mime, ThreadMetaProvider p,
        SaveOptions saveOptions, OnUpdateEnvelope hook ) const {
    syncInsertEntry( std::move(envelope), std::move(mime), p, std::move(saveOptions),
            [thiz = shared_from_this(), hook](error_code err, UpdateEnvelopeResult uer) {
        if(!err && std::get<EnvelopeKind>(uer) == EnvelopeKind::original) {
            const auto& e = std::get<Envelope>(uer);
            thiz->logOperation<params::CreateMessage>(id::state(e.mid()), id::affected(1ul), id::mid(e.mid()));
        }
        hook(err, std::move(uer));
    });
}


void EnvelopesRepository::updateInternal( const Envelope & old, Envelope newer, MimeParts mime,
        ThreadMetaProvider p, bool ignoreDuplicates, OnUpdateEnvelope hook ) const {
    syncUpdateEntry(old, newer, std::move(mime), p, ignoreDuplicates,
            [mid=old.mid(), thiz = shared_from_this(), hook] (
            error_code e, UpdateEnvelopeResult uer){
        if(!e && std::get<EnvelopeKind>(uer) == EnvelopeKind::original) {
            const auto& env = std::get<Envelope>(uer);
            thiz->logOperation<params::UpdateMessage>(id::state(mid + "->" + env.mid()),
                    id::affected(1ul), id::oldMid(mid), id::newMid(env.mid()));
        }
        hook(e, std::move(uer));
    });
}

namespace {

inline auto toVector(const std::list<std::string>& ids) {
    return std::vector<std::string>(ids.begin(), ids.end());
}

}

void EnvelopesRepository::removeInternal(const std::list<Mid>& mids, bool force, OnUpdate hook) const {
    ASSERT_BIGINT_ITEMS(mids);
    auto h = [thiz = shared_from_this(), mids, hook] (error_code ec, Revision revision) {
        if(!ec) {
            const std::string midsStr = boost::algorithm::join(mids, ",");
            thiz->logOperation<params::DeleteMessages>(id::state(midsStr),
                                                       id::affected(mids.size()), id::mids(toVector(mids)));
        }
        hook(ec, revision);
    };
    if (!mids.empty()) {
        if (force) {
            syncForceErase(mids, std::move(h));
        } else {
            syncErase(mids, std::move(h));
        }
    } else {
        hook();
    }
}

namespace {

inline auto getLids(const std::list<Label>& labels) {
    std::vector<Lid> lids;
    std::transform(labels.begin(), labels.end(), std::back_inserter(lids), boost::bind(&Label::lid, _1));
    return lids;
}

inline auto getLabelTypesAndSymbols(const std::list<Label>& labels) {
    std::vector<std::string> types;
    std::vector<std::string> symbols;
    for (const auto& l: labels) {
        types.push_back(l.type().title());
        symbols.push_back(l.symbolicName().title());
    }
    return std::make_tuple<id::labelTypes, id::labelSymbols>(
                id::labelTypes(types), id::labelSymbols(symbols));
}

}
void EnvelopesRepository::markEnvelopesInternal(const std::list<Mid>& ids,
        const std::list<Label>& labels, OnUpdate hook) const {
    ASSERT_NOT_EMPTY_ARG(ids);
    ASSERT_BIGINT_ITEMS(ids);
    ASSERT_NOT_EMPTY_ARG(labels);

    syncAddLabels(labels, ids,
        [thiz = shared_from_this(), ids, labels, hook] (error_code ec, Revision revision)
        {
            if(!ec) {
                const auto state = thiz->getUpdateLabelsState(Oper::label, IdsType::mids, ids, labels);
                thiz->logOperation<params::AddLabels>(id::state(state), id::affected(ids.size()),
                        id::mids(toVector(ids)), id::lids(getLids(labels)),
                        getLabelTypesAndSymbols(labels));
            }
            hook(ec, revision);
        }
    );
}

void EnvelopesRepository::markEnvelopesByThreadsInternal(const std::list<Tid>& threadIDs,
        const std::list<Label>& labels, OnUpdate hook) const {
    ASSERT_NOT_EMPTY_ARG(threadIDs);
    ASSERT_BIGINT_ITEMS(threadIDs);
    ASSERT_NOT_EMPTY_ARG(labels);

    syncAddLabelsByThreads(labels, threadIDs,
        [thiz = shared_from_this(), threadIDs, labels, hook] (error_code ec, UpdateMessagesResult result)
        {
            if(!ec) {
                const auto state = thiz->getUpdateLabelsState(Oper::label, IdsType::tids, threadIDs, labels);
                thiz->logOperation<params::AddLabelsByThreads>(id::state(state), id::affected(result.affected),
                        id::tids(toVector(threadIDs)), id::lids(getLids(labels)),
                        getLabelTypesAndSymbols(labels));
            }
            hook(ec, result.rev);
        }
    );
}

void EnvelopesRepository::unmarkEnvelopesInternal(const std::list<Mid>& ids,
        const std::list<Label>& labels, OnUpdate hook) const {
    ASSERT_NOT_EMPTY_ARG(ids);
    ASSERT_BIGINT_ITEMS(ids);
    ASSERT_NOT_EMPTY_ARG(labels);

    syncRemoveLabels(labels, ids,
        [thiz = shared_from_this(), ids, labels, hook] (error_code ec, Revision revision)
        {
            if(!ec) {
                const auto state = thiz->getUpdateLabelsState(Oper::unlabel, IdsType::mids, ids, labels);
                thiz->logOperation<params::RemoveLabels>(id::state(state), id::affected(ids.size()),
                        id::mids(toVector(ids)), id::lids(getLids(labels)),
                        getLabelTypesAndSymbols(labels));
            }
            hook(ec, revision);
        }
    );
}

void EnvelopesRepository::unmarkEnvelopesByThreadsInternal(const std::list<Tid>& threadIDs,
        const std::list<Label>& labels, OnUpdate hook) const {
    ASSERT_NOT_EMPTY_ARG(threadIDs);
    ASSERT_BIGINT_ITEMS(threadIDs);
    ASSERT_NOT_EMPTY_ARG(labels);

    syncRemoveLabelsByThreads(labels, threadIDs,
        [thiz = shared_from_this(), threadIDs, labels, hook] (error_code ec, UpdateMessagesResult result)
        {
            if(!ec) {
                const auto state = thiz->getUpdateLabelsState(Oper::unlabel, IdsType::tids, threadIDs, labels);
                thiz->logOperation<params::RemoveLabelsByThreads>(id::state(state), id::affected(result.affected),
                        id::tids(toVector(threadIDs)), id::lids(getLids(labels)),
                        getLabelTypesAndSymbols(labels));
            }
            hook(ec, result.rev);
        }
    );
}

void EnvelopesRepository::changeEnvelopesLabelsInternal(const std::list<Mid>& mids,
                                                        const std::list<Label>& labelsToAdd,
                                                        const std::list<Label>& labelsToDel,
                                                        OnUpdate hook) const {
    ASSERT_NOT_EMPTY_ARG(mids);
    ASSERT_BIGINT_ITEMS(mids);

    syncChangeLabels(mids, labelsToAdd, labelsToDel,
        [thiz = shared_from_this(), mids, labelsToAdd, labelsToDel, hook]
                  (error_code ec, Revision revision) {
            if(!ec) {
                if (!labelsToDel.empty()) {
                    const auto state = thiz->getUpdateLabelsState(Oper::unlabel, IdsType::mids, mids, labelsToDel);
                    thiz->logOperation<params::RemoveLabels>(id::state(state), id::affected(mids.size()),
                            id::mids(toVector(mids)), id::lids(getLids(labelsToDel)),
                            getLabelTypesAndSymbols(labelsToDel));
                }
                if (!labelsToAdd.empty()) {
                    const auto state = thiz->getUpdateLabelsState(Oper::label, IdsType::mids, mids, labelsToAdd);
                    thiz->logOperation<params::AddLabels>(id::state(state), id::affected(mids.size()),
                            id::mids(toVector(mids)), id::lids(getLids(labelsToAdd)),
                            getLabelTypesAndSymbols(labelsToAdd));
                }
            }
            hook(ec, revision);
        }
    );
}

/// get messages by thread
void EnvelopesRepository::listInThread(const EnvelopesSorting& order,
                                           const Tid& thread,
                                           const boost::variant<std::size_t, string> & message,
                                           const std::size_t count,
                                           OnEnvelopeReceive handler) const {
    ASSERT_BIGINT_ARG(thread);
    syncListInThread(order, thread, message, count, handler);
}

void EnvelopesRepository::listInThreadWithLabel(const Tid& threadId,
                                                const macs::Label& label,
                                                size_t from,
                                                std::size_t count,
                                                OnEnvelopeReceive handler) const {
    ASSERT_BIGINT_ARG(threadId);
    syncListInThreadWithLabel(threadId, label, from, count, handler);
}

void EnvelopesRepository::listInThreadWithoutLabel(const Tid& threadId,
                                                   const macs::Label& label,
                                                   size_t from,
                                                   std::size_t count,
                                                   OnEnvelopeReceive handler) const {
    ASSERT_BIGINT_ARG(threadId);
    syncListInThreadWithoutLabel(threadId, label, from, count, handler);
}

/// get messages for search
void EnvelopesRepository::listFilterSearch(bool unread,
        bool atta,
        const std::list<Fid>& fids,
        const std::list<Lid>& lids,
        const std::list<Mid>& mids,
        const std::string& folderSet,
        const EnvelopesSorting & order,
        OnEnvelopeReceive handler) const {
    ASSERT_NOT_EMPTY_ARG(mids);
    ASSERT_BIGINT_ITEMS(mids);
    ASSERT_INTEGER_ITEMS(fids);
    ASSERT_INTEGER_ITEMS(lids);
    syncListFilterSearch(unread, atta, fids, lids, mids, folderSet, order, handler);
}

void EnvelopesRepository::listWithNoAnswer(
        const RfcMessageId& msgId,
        const Lid& includeLid,
        OnEnvelopeReceive handler) const {
    ASSERT_INTEGER_ARG(includeLid);
    syncListWithNoAnswer(msgId, includeLid, handler);
}

void EnvelopesRepository::listInReplyTo(const RfcMessageId& msgId,
                                            OnEnvelopeReceive handler) const {
    syncListInReplyTo(msgId, handler);
}

void EnvelopesRepository::listInReplyToByMid(const Mid& mid,
                                                 OnEnvelopeReceive handler) const {
    ASSERT_BIGINT_ARG(mid);
    syncListInReplyToByMid(mid, handler);
}

void EnvelopesRepository::resetFreshCounterInternal(OnUpdate handler) const {
    syncResetFreshCounter([thiz = shared_from_this(), handler](error_code const & e, Revision r) {
        if(!e) {
            thiz->logOperation<params::ResetFresh>(id::state(""), id::affected(1ul), id::hidden(true));
        }
        handler(e, r);
    });
}

void EnvelopesRepository::moveMessagesInternal(
        const Fid& destFolder, const std::optional<Tab::Type>& destTab,
        const std::list<Mid>& mids, OnUpdate hook) const {
    ASSERT_NOT_EMPTY_ARG(mids);
    ASSERT_BIGINT_ITEMS(mids);
    syncMoveMessages(destFolder, destTab, mids,
        [thiz = shared_from_this(), destFolder, destTab, mids, hook] (const error_code &ec, Revision revision)
        {
            if(!ec) {
                std::ostringstream st;
                st << "fid=" << destFolder << ";mids=" << boost::algorithm::join(mids, ",");
                thiz->logOperation<params::MoveMessages>(id::state(st.str()), id::affected(mids.size()),
                        id::destFid(destFolder), id::mids(toVector(mids)));

                auto stringType = [](auto& t) -> std::string {
                    return t ? t->toString() : "";
                };
                st.clear();
                st << "tab=" << stringType(destTab) << ";mids=" << boost::algorithm::join(mids, ",");
                thiz->logOperation<params::MoveMessagesToTab>(id::state(st.str()), id::affected(mids.size()),
                        id::destTab(stringType(destTab)), id::mids(toVector(mids)));

            }
            hook(ec, revision);
        }
    );
}

void EnvelopesRepository::copyMessagesInternal(const Fid& dstFid,
        const std::vector<Mid>& mids, OnCopy hook) const {
    ASSERT_BIGINT_ITEMS(mids);
    ASSERT_INTEGER_ARG(dstFid);
    if (!mids.empty()) {
        const auto handle = [thiz = shared_from_this(), hook = std::move(hook), mids, dstFid]
                             (error_code ec, auto res) {
            if (!ec) {
                const auto midsStr = boost::algorithm::join(res.mids, ",");
                thiz->logOperation<params::CopyMessages>(id::state(midsStr),
                                        id::affected(mids.size()),
                                        id::destFid(dstFid), id::mids(mids));
            }
            hook(ec, res);
        };
        syncCopyMessages(dstFid, mids, handle);
    } else {
        hook();
    }
}

void EnvelopesRepository::updateStatusInternal(const std::list<Mid>& mids,
                                       Envelope::Status status,
                                       OnUpdate hook) const {
    ASSERT_BIGINT_ITEMS(mids);
    if (!mids.empty()) {
        syncUpdateStatus(mids, status, [thiz = shared_from_this(), mids,
                                        status, hook](error_code const & e, Revision r){
            if(!e) {
                const std::string midsStr = boost::algorithm::join(mids, ",");
                const std::string journalStatus = getJournalStatusString(status);
                thiz->logOperation<params::MarkMessages>(id::state(journalStatus + ";" + midsStr),
                        id::affected(mids.size()), id::mids(toVector(mids)), id::msgStatus(journalStatus));
            }
            hook(e,r);
        });
    } else {
        hook();
    }
}

void EnvelopesRepository::threadsByTids(const TidVector& tids,
                                        OnEnvelopeReceive handler) const {
    ASSERT_NOT_EMPTY_ARG(tids);
    ASSERT_BIGINT_ITEMS(tids);
    syncThreadsByTids(tids, handler);
}

void EnvelopesRepository::threadsWithLabel(const Lid& label,
                                               size_t from, size_t count,
                                               OnEnvelopeReceive handler) const {
    ASSERT_INTEGER_ARG(label);
    syncThreadsWithLabel(label, from, count, handler);
}

void EnvelopesRepository::threadsInFolderWithoutLabel(const Fid& folder,
                                                          const Lid& label,
                                                          size_t from,
                                                          size_t count,
                                                          OnEnvelopeReceive handler) const {
    ASSERT_INTEGER_ARG(folder);
    ASSERT_INTEGER_ARG(label);
    syncThreadsInFolderWithoutLabels(folder, {label}, from, count, handler);
}

void EnvelopesRepository::messagesInFolderWithoutLabel(const Fid& folder,
                                                           const Lid& label,
                                                           size_t from,
                                                           size_t count,
                                                           const EnvelopesSorting& order,
                                                           OnEnvelopeReceive handler) const {
    ASSERT_INTEGER_ARG(folder);
    ASSERT_INTEGER_ARG(label);
    syncMessagesInFolderWithoutLabels(folder, {label}, from, count, order, handler);
}

std::string EnvelopesRepository::getUpdateLabelsState(UserJournal::Operation operation,
            IdsType idsType, const std::list<std::string>& ids,
            const std::list<Label>& labels) const {
    const std::string opCode = operation == Oper::label ? ":=" : "!=";
    return toString(idsType) + "=" + boost::algorithm::join(ids, ",") + opCode
            + boost::algorithm::join(labels |
                    boost::adaptors::transformed(
                            boost::bind(&Label::lid, _1)), ",");
}

void EnvelopesRepository::envelopesQueryInMailbox(const Mid& fromMid,
                                                  size_t from, size_t count, bool groups,
                                                  const EnvelopesSorting& order,
                                                  const std::list<Lid>& labels,
                                                  const std::list<Lid>& excludeLabels,
                                                  OnEnvelopeReceive handler) const {
    syncEnvelopesQueryInMailbox(fromMid, from, count, groups,
                                order, labels, excludeLabels, handler);
}

void EnvelopesRepository::envelopesQueryInFolder(const EnvelopesQueryInFolder::Params& params,
                                                 OnEnvelopeReceive handler) const {

    syncEnvelopesQueryInFolder(params, handler);
}

void EnvelopesRepository::envelopesInFolderWithMimes(size_t from, size_t count, const EnvelopesSorting& order,
                                                  const Fid& fid, OnEnvelopeWithMimeReceive handler) const {
    ASSERT_INTEGER_ARG(fid);
    syncEnvelopesInFolderWithMimes(from, count, order, fid, handler);
}

void EnvelopesRepository::envelopesByMidsWithMimes(const MidList& mids, OnEnvelopeWithMimeReceive handler) const {
    ASSERT_BIGINT_ITEMS(mids);
    syncEnvelopesByMidsWithMimes(mids, handler);
}

void EnvelopesRepository::envelopesInFolderWithMimesByChunks(size_t minImapId, size_t maxImapId, size_t chunkSize,
                                                             const Fid& fid, OnEnvelopeWithMimeChunkReceive handler) const {
    ASSERT_INTEGER_ARG(fid);
    syncEnvelopesInFolderWithMimesByChunks(minImapId, maxImapId, chunkSize, fid, handler);
}

void EnvelopesRepository::getDeletedMessages(size_t from, size_t count,
                                             OnEnvelopeReceive handler) const {
    syncGetDeletedMessages(from, count, std::move(handler));
}

void EnvelopesRepository::getDeletedMessagesInInterval(size_t from, size_t count,
                                                       const std::pair<std::time_t, std::time_t>& timeInterval,
                                                       OnEnvelopeReceive handler) const {
    syncGetDeletedMessagesInInterval(from, count, timeInterval.first, timeInterval.second, std::move(handler));
}

void EnvelopesRepository::envelopesQueryInTab(const EnvelopesQueryInTab::Params& params,
                                              OnEnvelopeReceive handler) const {
    syncEnvelopesQueryInTab(params, handler);
}

}
