
#include <sstream>
#include <boost/unordered_map.hpp>
#include <boost/assign/list_of.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/range/algorithm/transform.hpp>
#include <boost/range/adaptors.hpp>
#include <macs/folder.h>

namespace macs {

const Folder::Symbol Folder::Symbol::inbox(SYM_INBOX, "inbox");
const Folder::Symbol Folder::Symbol::sent(SYM_SENT, "sent");
const Folder::Symbol Folder::Symbol::trash(SYM_TRASH, "trash");
const Folder::Symbol Folder::Symbol::spam(SYM_SPAM, "spam");
const Folder::Symbol Folder::Symbol::drafts(SYM_DRAFTS, "draft");
const Folder::Symbol Folder::Symbol::outbox(SYM_OUTBOX, "outbox");
const Folder::Symbol Folder::Symbol::archive(SYM_ARCHIVE, "archive");
const Folder::Symbol Folder::Symbol::template_(SYM_TEMPLATE, "template");
const Folder::Symbol Folder::Symbol::discount(SYM_DISCOUNT, "discount");
const Folder::Symbol Folder::Symbol::unsubscribe(SYM_UNSUBSCRIBE, "unsubscribe");
const Folder::Symbol Folder::Symbol::pending(SYM_PENDING, "pending");
const Folder::Symbol Folder::Symbol::hidden_trash(SYM_HIDDEN_TRASH, "hidden_trash");
const Folder::Symbol Folder::Symbol::restored(SYM_RESTORED, "restored");
const Folder::Symbol Folder::Symbol::reply_later(SYM_REPLY_LATER, "reply_later");

const Folder::Symbol Folder::Symbol::zombie_folder(SYM_ZOMBIE_FOLDER, "zombie");
const Folder::Symbol Folder::Symbol::none(SYM_FOLDER_NONE, "");

bool Folder::Symbol::isChangeble(const Folder::Symbol & symbol) {
    return symbol == Symbol::archive ||
            symbol == Symbol::discount ||
            symbol == Symbol::unsubscribe ||
            symbol == Symbol::template_;
}

//Hardcoded numbers from PL/SQL package "StoreLabels"
const Folder::Type Folder::Type::user(1,"user");
const Folder::Type Folder::Type::system(3,"system");

Folder::Path::Value Folder::Path::normalize(const Value& value) {
    Folder::Path::Value normalized;
    normalized.reserve(value.size());
    boost::transform(value, std::back_inserter(normalized), detail::normalizeName);

    return normalized;
}

std::string Folder::Path::toString() const {
    return joinFolderPath(value_);
}

const std::string Folder::noParent = "0";

const std::string pathSeparator = "|";

std::string getFolderDisplayName(const std::string& path) {
    auto nodes = parseFolderPath(path);
    return nodes.empty() ? "" : nodes.back();
}


namespace {

void verifyFolderName(const std::string& name) {
    if (name.empty()) {
        throw system_error(error_code(error::invalidArgument), "Folder name is empty");
    }
    if (name.length() > Folder::maxFolderNameLength()) {
        throw system_error(error_code(error::invalidArgument), "Folder name is too large");
    }
}
} // namespace

std::string normalizeAndVerifyFolderName(const std::string& rawName) {
    const auto name = detail::normalizeName(rawName);
    verifyFolderName(name);
    return name;
}

std::string escapeFolderName(const std::string& name, const std::string& separator) {
    return boost::replace_all_copy(name, separator, separator+separator);
}

std::vector<std::string> parseFolderPath(std::string path, const std::string& separator) {
    // We use BELL symbol here - it cannot be typed, it cannod be printed, it cannot be pass with url encode
    static const std::string unexistingSymbol = "\a";

    if (path.empty()) {
        return {};
    }

    std::vector<std::string> tokens;
    boost::replace_all(path, separator, unexistingSymbol);
    boost::replace_all(path, unexistingSymbol + unexistingSymbol, separator);
    boost::split(tokens, path, boost::is_any_of(unexistingSymbol));

    return tokens;
}

std::string joinFolderPath(const std::vector<std::string>& nodes, const std::string& separator) {
    if (nodes.empty()) {
        return "";
    }
    return boost::algorithm::join(
            nodes | boost::adaptors::transformed([](const auto& n){ return escapeFolderName(n); }),
            separator
    );
}

} // namespace macs

