#include <user_journal/parameters/folders.h>
#include <macs/folders_repository.h>

// boost
#include <boost/bind.hpp>
#include <boost/optional/optional.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#include <boost/range/algorithm/sort.hpp>
#include <boost/range/algorithm/set_algorithm.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <stdexcept>
#include <future>

#include <macs/method_caller.h>
#include <macs/detail/strutils.h>

#include <macs/data/folder_sort_options.h>
#include <macs/data/folder_position.h>

#include <boost/asio/coroutine.hpp>

using std::string;
using std::list;
using boost::bind;
using boost::ref;

namespace macs {

namespace params = user_journal::parameters;
namespace id = params::id;

FoldersRepository::CachePtr FoldersRepository::foldersInternal() const {
    std::promise<CachePtr> p;
    auto f = p.get_future();
    foldersInternal([&](error_code e, CachePtr v){
        auto res = std::move(p);
        if(e) {
            mail_errors::setSystemError(res, "foldersInternal", e);
        } else {
            res.set_value(v);
        }
    });
    return f.get();
}

void FoldersRepository::createFolderInternal(const std::string& rawName,
        const std::string& parent, const Folder::Symbol& symbol, OnUpdateFolder hook) const {

    const auto name = normalizeAndVerifyFolderName(rawName);
    foldersInternal([self = shared_from_this(), name, parent, symbol, hook = std::move(hook)]
                                     (error_code e, CachePtr folders) mutable {
        if(e) {
            hook(std::move(e));
            return;
        }

        e = folders->checkCanCreateFolder(name, parent, symbol != Folder::Symbol::none);
        if(e) {
            hook(std::move(e));
            return;
        }

        self->syncCreateFolder(name, parent, symbol, [self, hook = std::move(hook)]
                                        (error_code ec, Folder folder) {
            if (!ec) {
                self->onFolderCreate(std::move(folder), std::move(hook));
            } else {
                hook(std::move(ec), std::move(folder));
            }
        });
    });
}

void FoldersRepository::getOrCreateFolderInternal(const std::string& rawName,
        const std::string& parent, const Folder::Symbol& symbol, OnUpdateFolder hook) const {
    const auto name = normalizeAndVerifyFolderName(rawName);
    foldersInternal([self = shared_from_this(), name, parent, symbol, hook = std::move(hook)]
                    (error_code e, CachePtr folders) mutable {
        if(e) {
            hook(std::move(e));
            return;
        }

        e = folders->checkCanGetOrCreateFolder(parent, symbol != Folder::Symbol::none);
        if(e) {
            hook(std::move(e));
            return;
        }

        auto handler = [self, hook = std::move(hook)]
                       (error_code ec, Folder folder) mutable {
            if (!ec) {
                self->onFolderCreate(std::move(folder), std::move(hook));
            } else {
                hook(std::move(ec));
            }
        };

        self->syncGetOrCreateFolder(name, parent, symbol, std::move(handler));
    });
}

void FoldersRepository::createFolderByPathInternal(const Folder::Path& path, OnUpdateFolder hook) const {
    if (path.empty()) {
        throw system_error(error_code(error::invalidArgument), "createFolderByPathInternal() path is empty");
    }

    syncCreateFolderByPath(path,
        [self = shared_from_this(), hook = std::move(hook), path] (error_code ec, Folder folder) {
            if (ec) {
                hook(std::move(ec));
                return;
            }

            self->logOperation<params::CreateFolder>(id::state(folder.name()), id::affected(0ul),
                    id::fid(folder.fid()),
                    id::parentFid(folder.parentId()),
                    id::folderName(folder.name()),
                    id::folderType(folder.type().title()),
                    id::folderSymbol(folder.symbolicName().title()));
            self->resetFoldersCache();
            hook(std::move(folder));
        });
}

std::vector<Folder> FoldersRepository::setFoldersInitialPosition(FoldersMap & f) const {
    auto folders = *foldersInternal();
    std::vector<Folder> retval;
    for (auto & i : f) {
        const auto it = folders.find(i.second.fid());
        if(it == folders.end()) {
            throw std::logic_error("setFoldersInitialPosition: bad fid " + i.second.fid());
        }
        Folder folder = it->second;
        const auto old = folder.position();
        setFolderPosition(folder, folders);
        if (old != folder.position()) {
            retval.push_back(folder);
        }
    }
    for (const auto& folder : retval) {
        f.erase(folder.fid());
        f.insert({folder.fid(), folder});
    }
    return retval;
}


#include <boost/asio/yield.hpp>

using SelfPtr = std::shared_ptr<const macs::FoldersRepository>;

template <typename Handler>
void FoldersRepository::updatePositionForFolders(std::vector<Folder> folders, Handler h) const {
    struct PositionUpdater : boost::asio::coroutine {
        struct Ctx {
            std::vector<Folder> folders;
            SelfPtr self;
            Handler h;
        };

        std::shared_ptr<Ctx> ctx;

        PositionUpdater(std::vector<Folder> folders, SelfPtr self, Handler h)
        : ctx(std::make_shared<Ctx>(Ctx{std::move(folders), self, std::move(h)})) {}

        void operator() (error_code e = error_code(), Folder = Folder()) {
            reenter(*this) {
                while(!ctx->folders.empty() && !e) {
                    yield {
                        auto folder = std::move(ctx->folders.back());
                        ctx->folders.pop_back();
                        ctx->self->updateFolder(std::move(folder), *this);
                    };
                }
                ctx->h(e, ctx->self);
            }
        }
    } coro { std::move(folders), shared_from_this(), std::move(h) };
    coro();
}

#include <boost/asio/unyield.hpp>

void FoldersRepository::onFolderCreate(Folder folder, OnUpdateFolder hook) const {
    logOperation<params::CreateFolder>(id::state(folder.name()),
                                       id::affected(0ul),
                                       id::fid(folder.fid()),
                                       id::parentFid(folder.parentId()),
                                       id::folderName(folder.name()),
                                       id::folderType(folder.type().title()),
                                       id::folderSymbol(folder.symbolicName().title()));
    const auto old = folder.position();
    resetFoldersCache();
    foldersInternal([self = shared_from_this(), hook = std::move(hook), old, f = std::move(folder)]
                    (error_code e, CachePtr v) mutable {
        if (!e) {
            self->setFolderPosition(f, *v);
            if( old != f.position() ) {
                return self->updatePositionInternal(f, hook);
            }
        }
        hook(std::move(e), std::move(f));
    });
}

void FoldersRepository::subscribeToSharedFoldersInternal(
        const std::string& sharedFoldersSuid, OnFoldersMap handler) const {

    syncSubscribeToSharedFolders(sharedFoldersSuid,
            [self = shared_from_this(), handler = std::move(handler)]
                (error_code e, FoldersMap subscribed) mutable {
        if(!e) {
            auto folders = self->setFoldersInitialPosition(subscribed);
            self->updatePositionForFolders(std::move(folders),
                    [subscribed = std::move(subscribed), handler = std::move(handler)]
                     (error_code e, SelfPtr self) mutable{
                self->resetFoldersCache();
                handler(std::move(e), std::move(subscribed));
            });
        } else {
            handler(e, std::move(subscribed));
        }
    });
}

void FoldersRepository::updateFolderInternal(Folder folder, OnUpdateFolder hook) const {
    const auto& defaults = defaultFoldersSymbols();
    foldersInternal([self = shared_from_this(),
                     folder = std::move(folder),
                     hook = std::move(hook), &defaults] (error_code e, CachePtr fs) mutable {

        if (e) {
            return hook(std::move(e));
        }

        const auto i = fs->find(folder.fid());
        if (i == fs->end()) {
            return hook(error_code(error::noSuchFolder, "can't update folder "
                    + folder.fid()), Folder());
        }

        const Folder& original = i->second;
        const bool nameChanged = folder.name() != original.name();
        if (nameChanged) {
            e = fs->checkCanRenameFolder(folder, original);
            if(e) {
                return hook(std::move(e));
            }
        }

        const bool parentChanged = folder.parentId() != original.parentId();
        if (parentChanged) {
            e = fs->checkCanMoveFolder(folder, original);
            if(e) {
                return hook(std::move(e));
            }
            self->setFolderPosition(folder, *fs);
        }

        const bool positionChanged = folder.position() != original.position();
        if (nameChanged || parentChanged) {

            if (defaults.contains(original.symbolicName())) {
                return hook(error_code(error::cantModifyFolder, "can't update folder "
                + folder.fid() + ", it is default"), folder);
            }

            self->syncModifyFolder(folder,
                    [self, positionChanged, hook = std::move(hook)]
                                  (error_code ec, Folder f) mutable {
                if (!ec) {
                    self->resetFoldersCache();
                    self->logOperation<params::RenameFolder>(id::state(f.fid()), id::affected(0ul),
                            id::fid(f.fid()), id::parentFid(f.parentId()), id::folderName(f.name()));
                    if (positionChanged) {
                        return self->updatePositionInternal(std::move(f), std::move(hook));
                    }
                }
                hook(std::move(ec), std::move(f));
            });
        } else if (positionChanged) {
            self->updatePositionInternal(std::move(folder), std::move(hook));
        } else {
            hook();
        }
    });
}

void FoldersRepository::updateFolderToPathInternal(FolderFactory factory,
                                                   Folder::Path path,
                                                   OnUpdateFolder hook) const {
    foldersInternal([self = shared_from_this(),
                     factory = std::move(factory),
                     path = std::move(path),
                     hook = std::move(hook)] (error_code e, CachePtr fs) mutable {

        if(e) {
            return hook(std::move(e));
        }

        factory.name(path.name());
        Folder folder = factory.product();

        const auto i = fs->find(folder.fid());
        if (i == fs->end()) {
            return hook(error_code(error::noSuchFolder, "can't update folder "
                    + folder.fid()), Folder());
        }
        const Folder & original = i->second;

        e = fs->checkCanUpdateFolder(folder, original);
        if (e) {
            return hook(std::move(e));
        }

        Folder::Path originalPath = fs->getPath(original);
        const bool pathChanged = (path.size() != originalPath.size()) ||
                                 !std::equal(path.begin(), path.end(), originalPath.begin());
        const bool positionChanged = folder.position() != original.position();

        if (pathChanged) {
            self->syncModifyFolderToPath(folder, path,
                    [self, positionChanged, hook = std::move(hook)]
                                  (error_code ec, Folder f) {
                if (!ec) {
                    self->resetFoldersCache();
                    self->logOperation<params::RenameFolder>(id::state(f.fid()), id::affected(0ul),
                            id::fid(f.fid()), id::parentFid(f.parentId()), id::folderName(f.name()));
                    if (positionChanged) {
                        return self->updatePositionInternal(std::move(f), std::move(hook));
                    }
                }
                hook(std::move(ec), std::move(f));
            });
        } else if (positionChanged) {
            self->updatePositionInternal(std::move(folder), std::move(hook));
        } else {
            hook();
        }
    });
}

void FoldersRepository::updatePositionInternal(Folder folder, OnUpdateFolder handler) const {
    syncSetPosition(folder.fid(), folder.position(),
            [thiz = shared_from_this(), folder, handler](error_code ec, Revision rev){
                Folder updatedFolder = FolderFactory(std::move(folder)).revision(rev).product();
                thiz->resetFoldersCache();
                handler(std::move(ec), std::move(updatedFolder));
            });
}

void FoldersRepository::eraseInternal(const Folder & folder, OnUpdate handler) const {
    foldersInternal([self = shared_from_this(), folder,
                     handler = std::move(handler)]
                             (error_code e, CachePtr folders) mutable {
        if(e) {
            handler(e, NULL_REVISION);
            return;
        }

        e = folders->checkCanEraseFolder(folder);
        if(e) {
            handler(e, NULL_REVISION);
            return;
        }

        const auto fid = folder.fid();
        self->syncEraseFolder(fid, [self, fid, handler = std::move(handler)]
                         (error_code ec, Revision revision) {
            if (!ec) {
                self->logOperation<params::DeleteFolder>(id::state(fid), id::affected(0ul), id::fid(fid));
                self->resetFoldersCache();
            }
            handler(ec, revision);
        });
    });
}

void FoldersRepository::eraseCascadeInternal(const Folder & folder, OnUpdate handler) const {
    const auto e = checkCanEraseFolderCascade(folder);
    if(e) {
        handler(e, NULL_REVISION);
        return;
    }

    const auto fid = folder.fid();

    getAllFolders([self = shared_from_this(), fid, handler = std::move(handler)]
                     (error_code e, FolderSet folders) mutable {
        if(e) {
            handler(e, NULL_REVISION);
            return;
        }

        self->syncClearFolderCascade(fid, folders, [self, fid, handler = std::move(handler)]
               (error_code ec, auto res) {
            if (!ec) {
                self->logOperation<params::DeleteFolder>(id::state(fid),
                        id::affected(res.affected), id::fid(fid));
                self->resetFoldersCache();
            }
            handler(ec, res.rev);
        });
    });
}

void FoldersRepository::moveAssociatedDataInternal(const Folder & from, const Folder & to,
        OnUpdate handler) const {
    syncMoveAll(from, to, [self = shared_from_this(), from, to, handler]
            ( error_code ec, auto res) {
        if (!ec) {
            self->logOperation<params::MoveFolderMessages>(id::state(from.fid() + "->" + to.fid()),
                    id::affected(res.affected), id::srcFid(from.fid()), id::destFid(to.fid()));
            self->resetFoldersCache();
        }
        handler(ec, res.rev);
    });
}

Folder FoldersRepository::getFolderByFid(const string& fid) const {
    return foldersInternal()->at(fid);
}

string FoldersRepository::getFolderFidBySymbol(const Folder::Symbol& symbol) const {
    return symbol == Folder::Symbol::none ? string() : foldersInternal()->fid(symbol);
}

Folder FoldersRepository::getFolderBySymbol(const Folder::Symbol & symbol) const {
    return foldersInternal()->at(symbol);
}

void FoldersRepository::setSymbolInternal(const Folder & folder, const Folder::Symbol & symbol,
        OnUpdate handler) const {

    const auto fid = folder.fid();
    if (!Folder::Symbol::isChangeble(symbol)) {
        handler(error_code(error::cantModifyFolder,
                "can't change symbol to " + symbol.title() + " for fid: " + fid),
                NULL_REVISION);
        return;
    }

    if (folder.symbolicName() != Folder::Symbol::none) {
        handler(error_code(error::cantModifyFolder,
                "can't change symbol name for fid: " + fid +
                ": symbol already exist " + folder.symbolicName().title()),
                NULL_REVISION);
        return;
    }

    syncSetFolderSymbol(fid, symbol,
            [self = shared_from_this(), handler = std::move(handler), symbol, fid]
                                      (error_code e, Revision r) {
        if(!e) {
            self->logOperation<params::ChangeFolderSymbol>(id::state(fid + "+" + symbol.title()),
                    id::affected(0ul), id::fid(fid), id::folderSymbol(symbol.title()));
            self->resetFoldersCache();
        }
        handler(e, r);
    });
}

void FoldersRepository::resetSymbolInternal(const Folder & folder, OnUpdate handler) const {

    const auto fid = folder.fid();

    if (folder.symbolicName() == Folder::Symbol::none) {
        handler(error_code(error::cantModifyFolder,
                "can't change symbol name for fid: " + fid +
                ": folder don't have symbol"),
                NULL_REVISION);
        return;
    }

    if (!folder.isChangeble()) {
        handler(error_code(error::cantModifyFolder,
                            "can't change symbol name for fid: " + fid +
                            ": can't change system folder"),
                            NULL_REVISION);
        return;
    }

    syncSetFolderSymbol(fid, Folder::Symbol::none,
            [self = shared_from_this(), handler = std::move(handler), fid]
             (error_code e, Revision r) {
        if(!e) {
            self->logOperation<params::ChangeFolderSymbol>(id::state(fid), id::affected(0ul), id::fid(fid));
            self->resetFoldersCache();
        }
        handler(e, r);
    });
}

bool FoldersRepository::existFolder(const string& fid) const {
    return foldersInternal()->exists(fid);
}

void FoldersRepository::resetUnvisitedInternal(const Folder & folder, OnUpdate hook) const {
    if (folder.unvisited()) {
        syncResetUnvisited(folder.fid(),
                [self = shared_from_this(), hook = std::move(hook)]
                (error_code ec, Revision revision) {
            if (!ec) {
                self->resetFoldersCache();
            }
            hook(std::move(ec), revision);
        });
    } else {
        hook(error_code(), NULL_REVISION);
    }
}

void FoldersRepository::setFolderPosition(Folder& folder, const FolderSet& fs) const {
    const size_t position = GetFolderPosition(fs)(folder.parentId());
    FolderFactory factory(std::move(folder));
    factory.position(position);
    folder = factory.product();
}

const Folder::SymbolSet& FoldersRepository::defaultFoldersSymbols() const {
    static Folder::SymbolSet result = {
        Folder::Symbol::inbox,
        Folder::Symbol::drafts,
        Folder::Symbol::sent,
        Folder::Symbol::trash,
        Folder::Symbol::spam,
        Folder::Symbol::outbox,
    };
    return result;
}

error_code FoldersRepository::checkFoldersRange(const FolderSet& folders) const {
    using It = Folder::SymbolSet::const_iterator;

    const auto& defaults = defaultFoldersSymbols();
    std::vector<It> foundSymbols;

    foundSymbols.reserve(defaults.size());

    for (const auto& i : folders) {
        const auto it = defaults.find(i.second.symbolicName());
        if (it != defaults.end()) {
            foundSymbols.push_back(it);
        }
    }

    if (foundSymbols.size() != defaults.size()) {
        using boost::algorithm::join;
        using boost::adaptors::transformed;

        boost::sort(foundSymbols, [] (It lhs, It rhs) { return *lhs < *rhs; });

        const auto values = foundSymbols | transformed([] (It x) { return *x; });

        std::vector<Folder::Symbol> missing;
        boost::set_difference(defaults, values, std::back_inserter(missing));

        std::vector<Folder::Symbol> duplicates;
        boost::set_difference(values, defaults, std::back_inserter(duplicates));

        auto toTitle = transformed([](const Folder::Symbol& x) { return x.title(); });

        std::vector<std::string> errors;

        if (!missing.empty()) {
            errors.push_back("doesn't contains folders with symbols=" + join(missing | toTitle, ","));
        }

        if (!duplicates.empty()) {
            errors.push_back("contains folders with duplicate symbols=" + join(duplicates | toTitle, ","));
        }


        return error_code(error::userNotInitialized, "folder list " + join(errors, " and "));
    }
    return error_code();
}

void FoldersRepository::setPop3Internal(std::vector<std::string> fids, OnUpdate handler) const {
    const auto u = std::unique(fids.begin(), fids.end());
    const auto newSize = static_cast<size_t>(std::distance(fids.begin(), u));
    fids.resize(newSize);

    getAllFolders([self = shared_from_this(), fids = std::move(fids),
                   handler = std::move(handler)]
                         (error_code e, FolderSet fs) mutable {
        if(e) {
            handler(std::move(e), NULL_REVISION);
            return;
        }

        if (!fids.empty()) {
            const auto i = std::find_if(fids.begin(), fids.end(),
                    [&](const std::string& fid){ return !fs.exists(fid); });
            if( i != fids.end()) {
                handler(error_code(error::noSuchFolder,
                        "setPop3(): folder does not exist fid=" + *i), NULL_REVISION);
                return;
            }
        }

        self->syncSetPop3(fids, fs, [self, fids, handler] (error_code ec, Revision revision) {
            if (!ec) {
                self->logOperation<params::SetPop3>(id::state(boost::algorithm::join(fids, ", ")),
                        id::affected(0ul), id::fids(fids));
                self->resetFoldersCache();
            }
            handler(std::move(ec), revision);
        });
    });
}

} // namespace macs
