#include <user_journal/parameters/labels.h>
#include <macs/labels_repository.h>

#include <stdexcept>
#include <future>

#include <mail_errors/set_exception.h>
#include <macs/detail/strutils.h>

using std::string;
using std::list;

namespace macs {

namespace params = user_journal::parameters;
namespace id = params::id;

LabelsRepository::CachePtr LabelsRepository::labelsInternal() const {
    std::promise<CachePtr> p;
    auto f = p.get_future();
    labelsInternal([&](error_code e, CachePtr v){
        auto res = std::move(p);
        if(e) {
            mail_errors::setSystemError(res, "labelsInternal", e);
        } else {
            res.set_value(std::move(v));
        }
    });
    return f.get();
}

void LabelsRepository::createLabelWithTypeInternal(const std::string& rawName,
                                           const std::string& color,
                                           const std::string& type,
                                           OnUpdateLabel hook) const {
    const std::string name = detail::normalizeName(rawName);

    if(name.empty()) {
        hook(error_code(error::invalidArgument,
                "can't create label with no name"));
        return;
    }

    if(name.length() > Label::maxLabelNameLength()) {
        hook(error_code(error::invalidArgument, "Label name too long"));
        return;
    }

    const Label::Type labelType = Label::Type::getByTitle(type);
    labelsInternal([self = shared_from_this(), h = std::move(hook), name, color, labelType]
                    (error_code e, CachePtr labels) {

        if(e) {
            h(std::move(e));
            return;
        }

        if (labels->exists(name, labelType)) {
            std::ostringstream s;
            s << "can't create label: already exist label with name " << name << " type " << labelType.title();
            h(error_code(error::duplicateLabelNameType, s.str()));
            return;
        }

        self->syncCreateLabel(name, color, labelType,
                [self, h = std::move(h), labelType, name]
                 (error_code ec, Label lbl) {
            if (!ec) {
                self->logOnCreateLabel(lbl);
                self->resetLabelsCache();
            }
            h(std::move(ec), std::move(lbl));
        });

    });

}

void LabelsRepository::createLabelInternal(const Label::Symbol & symbol, OnUpdateLabel hook) const {

    labelsInternal([self = shared_from_this(), h = std::move(hook), symbol]
                    (error_code e, CachePtr labels) {

        if(e) {
            h(std::move(e));
            return;
        }

        if (labels->exists(symbol)) {
            std::ostringstream s;
            s << "can't create label: already exist label with symbol " << symbol.title();
            h(error_code(error::duplicateLabelSymbol, s.str()));
            return;
        }

        self->syncCreateLabel(symbol, [self, h = std::move(h)] (error_code ec, Label lbl) {
            if (!ec) {
                self->logOnCreateLabel(lbl);
                self->resetLabelsCache();
            }
            h(std::move(ec), std::move(lbl));
        });

    });
}

void LabelsRepository::updateLabelInternal(Label label, OnUpdateLabel hook) const {
    labelsInternal(
            [self = shared_from_this(), h = std::move(hook), label = std::move(label)]
            (error_code e, CachePtr labels) {

        if(e) {
            h(std::move(e));
            return;
        }

        const auto it = labels->find(label.lid());
        if (it == labels->end()) {
            std::ostringstream s;
            s << "can't update label with lid '" << label.lid() << "'";
            h(error_code(error::noSuchLabel, s.str()));
            return;
        }

        const auto & original = it->second;
        const bool nameChanged = label.name() != original.name();
        const bool colorChanged = label.color() != original.color();

        if (!(nameChanged || colorChanged)) {
            h(original);
            return;
        }

        if (nameChanged) {
            std::ostringstream s;
            if (label.name().empty()) {
                s << "can't rename label " << label.lid() << " to empty name";
                h(error_code(error::invalidArgument, s.str()));
                return;
            }
            if (label.name().length() > Label::maxLabelNameLength()) {
                s << "can't rename label " << label.lid() << ": too long name";
                h(error_code(error::invalidArgument, s.str()));
                return;
            }
            if (!original.isUser()) {
                s << "can't rename non-user label " << label.lid();
                h(error_code(error::cantModifyLabel, s.str()));
                return;
            }
        }

        if (colorChanged) {
            std::ostringstream s;
            if (label.color().empty()) {
                s << "can't change color to empty for label " << label.lid();
                h(error_code(error::invalidArgument, s.str()));
                return;
            }
            if (!original.isUser()) {
                s << "can't change color for non-user label " << label.lid();
                h(error_code(error::invalidArgument, s.str()));
                return;
            }
        }

        self->syncModifyLabel(label, [self, h = std::move(h)]
                              (error_code ec, Label label){
            if (!ec) {
                self->logOperation<params::RenameLabel>(id::state(label.lid()),
                        id::affected(0ul),
                        id::lid(label.lid()),
                        id::labelName(label.name()),
                        id::labelColor(label.color()));
                self->resetLabelsCache();
            }
            h(std::move(ec), std::move(label));
        });
    });
}


void LabelsRepository::deleteLabelInternal(const std::string& lid, OnUpdate hook) const {
    if(lid.empty()) {
        hook(error_code(error::invalidArgument,
                "can't delete label with empty lid"), NULL_REVISION);
        return;
    }

    labelsInternal(
                [self = shared_from_this(), h = std::move(hook), lid]
                (error_code e, CachePtr labels) mutable {

        if(e) {
            h(e, NULL_REVISION);
            return;
        }

        const auto it = labels->find(lid);
        if (it == labels->end()) {
            std::ostringstream s;
            s << "can't delete label lid '" << lid << "'";
            h(error_code(error::noSuchLabel, s.str()), NULL_REVISION);
            return;
        }

        const auto & type = it->second.type();
        if( type == Label::Type::system || type == Label::Type::threadWide) {
            std::ostringstream s;
            s << "can't delete label " << lid << " with type '" << type.title() << "'";
            h(error_code(error::cantModifyLabel, s.str()), NULL_REVISION);
            return;
        }

        self->syncEraseLabel(lid, [self, lid, h = std::move(h)]
               (error_code ec, Revision revision) {

           if (!ec) {
               self->logOperation<params::DeleteLabel>(id::state(lid), id::affected(0ul), id::lid(lid));
               self->resetLabelsCache();
           }
           h(ec, revision);
       });
    });
}

Label LabelsRepository::getLabelByLid(const string& lid) const {
    return labelsInternal()->at(lid);
}

std::string LabelsRepository::getLabelLidBySymbol(const Label::Symbol& symbol) const {
    return symbol == Label::Symbol::none ? LabelSet::null() : labelsInternal()->lid(symbol);
}

Label LabelsRepository::getLabelBySymbol(const Label::Symbol & symbol) const {
    return labelsInternal()->at(symbol);
}

bool LabelsRepository::existLabel(const string& lid) const {
    return labelsInternal()->exists(lid);
}

bool LabelsRepository::existLabel(const Label::Symbol & symbol) const {
    return labelsInternal()->exists(symbol);
}

bool LabelsRepository::existLabelNameAndType(const std::string & name, const Label::Type & type) const {
    return labelsInternal()->exists(name, type);
}
void LabelsRepository::clearLabelInternal(const Lid& lid, OnUpdateMessages hook) const {
    if(lid.empty()) {
        hook(error_code(), {Revision(), 0});
        return;
    }

    labelsInternal(
                [self = shared_from_this(), h = std::move(hook), lid]
                (error_code e, CachePtr labels) mutable {

        if(e) {
            h(e, {Revision(), 0});
            return;
        }

        const auto i = labels->find(lid);

        if (i == labels->end()) {
            h(error_code(), {Revision(), 0});
            return;
        }

        const auto & label = i->second;
        self->syncClearLabel(label, [self, h = std::move(h), lid]
                 (error_code e, auto res) {
             if (!e) {
                 self->logOperation<params::ClearLabel>(id::state(lid), id::affected(0ul), id::lid(lid));
                 self->resetLabelsCache();
             }
             h(e, res);
         });
    });
}

std::string LabelsRepository::getLabelLidByNameAndType(std::string const& name, const Label::Type & type) const {
    return labelsInternal()->lid(name, type);
}

Label LabelsRepository::getLabelByNameAndType(const std::string& name, const Label::Type & type) const {
    return labelsInternal()->at(name, type);
}

void LabelsRepository::getOrCreateLabelInternal(const std::string& rawName,
                                           const std::string& color,
                                           const Label::Type& type,
                                           OnUpdateLabel hook) const {

    const std::string name = detail::normalizeName(rawName);

    if(name.empty()) {
        hook(error_code(error::invalidArgument,
                "can't create label with no name"));
        return;
    }

    if(name.length() > Label::maxLabelNameLength()) {
        hook(error_code(error::invalidArgument, "Label name too long"));
        return;
    }

    if (auto labels = cache.get()) {
        const auto i = labels->find(name, type);
        if (i != labels->end()) {
            hook(i->second);
            return;
        }

    }

    syncGetOrCreateLabel(name, color, type,
            [self = shared_from_this(), type, name, hook = std::move(hook)]
             (error_code ec, Label label) {
        if (!ec) {
            self->logOnCreateLabel(label);
            self->resetLabelsCache();
        }
        hook(std::move(ec), std::move(label));
    });
}

void LabelsRepository::getOrCreateLabelInternal(const Label::Symbol & symbol, OnUpdateLabel hook) const {

    if (auto labels = cache.get()) {
        const auto i = labels->find(symbol);
        if (i != labels->end()) {
            hook(i->second);
            return;
        }

    }

    syncGetOrCreateLabel(symbol,
            [self = shared_from_this(), hook = std::move(hook)]
             (error_code ec, Label label) {
        if (!ec) {
            self->logOnCreateLabel(label);
            self->resetLabelsCache();
        }
        hook(std::move(ec), std::move(label));
    });
}

void LabelsRepository::logOnCreateLabel(const Label &label) const {
    const bool needToHide = label.type() == Label::Type::system;
    logOperation<params::CreateLabel>(
            id::state(label.name()),
                id::affected(0ul),
                id::hidden(needToHide),
                id::lid(label.lid()),
                id::labelName(label.name()),
                id::labelType(label.type().title()),
                id::labelSymbol(label.symbolicName().title()),
                id::labelColor(label.color())
    );
}

} // namespace macs
