#include <macs/settings_init_values.h>
#include <macs/settings_description.h>

#include <boost/range/algorithm.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/lexical_cast/try_lexical_convert.hpp>

namespace macs::settings {

Description::Description() {
    initProductionProfile = getInitValues(production.profile);
    initProductionParameters = getInitValues(production.parameters);
    initProductionSettings.insert(initProductionProfile.begin(), initProductionProfile.end());
    initProductionSettings.insert(initProductionParameters.begin(), initProductionParameters.end());

    initCorpProfile = getInitValues(corp.profile);
    initCorpParameters = getInitValues(corp.parameters);
    initCorpSettings.insert(initCorpProfile.begin(), initCorpProfile.end());
    initCorpSettings.insert(initCorpParameters.begin(), initCorpParameters.end());
}

SettingsMap Description::getProfile(const SettingsMap& settings) const {
    const auto& profile = getRestrictions().profile;
    auto result = settings
        | boost::adaptors::filtered([&] (const auto& v) {
            return profile.find(v.first) != profile.end();
        });
    return {result.begin(), result.end()};
}

SettingsMap Description::getParameters(const SettingsMap& settings) const {
    const auto& profile = getRestrictions().profile;
    auto result = settings
        | boost::adaptors::filtered([&] (const auto& v) {
            return profile.find(v.first) == profile.end();
        });
    return {result.begin(), result.end()};
}

bool Description::isParameters(const SettingsMap& settings) const {
    const auto& profile = getRestrictions().profile;
    const auto it = boost::range::find_if(
        settings,
        [&] (const auto& v) {
            return profile.find(v.first) != profile.end();
        }
    );
    return it == settings.end();
}

bool Description::isParameters(const SettingsList& settingNames) const {
    const auto& profile = getRestrictions().profile;
    const auto it = boost::range::find_if(
        settingNames,
        [&] (const auto& v) {
            return profile.find(v) != profile.end();
        }
    );
    return it == settingNames.end();
}

bool Description::isProtectedParameters(const SettingsMap& settings) const {
    const auto& parameters = getRestrictions().parameters;
    const auto it = boost::range::find_if(
        settings,
        [&] (const auto& v) {
            const auto& param_it = parameters.find(v.first);
            return param_it != parameters.end() && param_it->second.updatability == Updatability::defended;
        }
    );
    return it != settings.end();
}

SettingsMap Description::validateProfile(const SettingsMap& settings) const {
    const auto& profile = getRestrictions().profile;
    auto result = settings
        | boost::adaptors::transformed([&] (const auto& v) {
            return make_pair(profile.find(v.first), std::cref(v));

        })
        | boost::adaptors::filtered([&] (const auto& v) {
            return v.first != profile.end() && v.first->second.updatability == Updatability::updated;
        })
        | boost::adaptors::transformed([&] (const auto& v) {
            return validateSetting(v);
        });
    return {result.begin(), result.end()};
}

SettingsMap::value_type Description::validateSetting(
        const std::pair<Restrictions::const_iterator, SettingsMap::value_type>& v) const {
    const auto& it = v.first;
    const auto& value = v.second;

    if (const auto& onInit = it->second.onInit; !onInit) {
        return value;
    } else if (const auto& type = it->second.type; type != Type::no_type) {
        return validateFlag(value, it, *onInit, type);
    } else if (const auto& limits = it->second.limits; limits) {
        return validateLimits(value, *limits, *onInit);
    } else if (const auto& values = it->second.values; !values.empty()) {
        return validateValues(value, values, *onInit);
    }
    return value;
}

SettingsMap Description::getInitValues(const std::unordered_map<std::string, Restriction>& description) {
    auto result = description
        | boost::adaptors::filtered([] (const auto& v) {
            return v.second.onInit.has_value();
        })
        | boost::adaptors::transformed([] (const auto& v) {
            return std::make_pair(v.first, *v.second.onInit);
        });
    return {result.begin(), result.end()};
}

SettingsMap::value_type Description::validateFlag(
        const SettingsMap::value_type& value,
        const Restrictions::const_iterator& it,
        const std::string& onInit,
        Type type) const {
    if (type == Type::flag) {
        return std::make_pair(value.first, setFlag(value.second));
    }
    if (const auto& limits = it->second.limits; type == Type::chr && limits) {
        if (std::int32_t c = value.second[0]; value.second.size() != 1
            || c > limits->max
            || c < limits->min
        ) {
            return std::make_pair(value.first, onInit);
        }
    }
    return value;
}

SettingsMap::value_type Description::validateLimits(
        const SettingsMap::value_type& value,
        const Limits& limits,
        const std::string& onInit) const {
    if (std::int32_t setting; !boost::conversion::try_lexical_convert(value.second, setting)
            || setting > limits.max
            || setting < limits.min) {
        return std::make_pair(value.first, onInit);
    }
    return value;
}

SettingsMap::value_type Description::validateValues(
        const SettingsMap::value_type& value,
        const Values& values,
        const std::string& onInit) const {
    if (values.find(value.second) == values.end()) {
        return std::make_pair(value.first, onInit);
    }
    return value;
}

}
