#include <user_journal/parameters/stickers.h>
#include <macs/stickers_repository.h>

namespace macs {

namespace params = user_journal::parameters;
namespace id = params::id;


void StickersRepository::checkDateInInterval(std::time_t date, std::time_t now, const StickerAllowedInterval& allowedInterval) {
    const std::time_t interval = date - now;
    if (interval < allowedInterval.min || interval > allowedInterval.max) {
        throw system_error(error_code(error::stickerDateOutOfInterval));
    }
}

void StickersRepository::createReplyLaterInternal(
        const Mid& mid, 
        const Fid& fid, 
        std::time_t date, 
        const StickerAllowedInterval& allowedInterval, 
        const std::optional<Tab::Type>& tab,
        OnReplyLaterSticker hook ) const {
    
    
    checkDateInInterval(date, std::time(nullptr), allowedInterval);
    
    asyncCreateReplyLater(mid, fid, date, tab,
            [self = shared_from_this(), hook = std::move(hook)]
            (error_code ec, ReplyLaterSticker sticker) {
                if(!ec) {
                    self->logOperation<params::CreateReplyLaterSticker>(
                        id::state(sticker.mid),
                        id::affected(1ul),
                        id::mid(sticker.mid),
                        id::fid(sticker.fid),
                        id::tab(sticker.tab ? sticker.tab->toString() : ""),
                        id::replyLaterDate(sticker.date),
                        id::stickerSymbol(yamail::data::reflection::to_string(macs::StickerType::reply_later)));
                }
                hook(std::move(ec), std::move(sticker));
            });
}

void StickersRepository::removeReplyLaterInternal(const Mid& mid, OnUpdate hook) const {
    asyncRemoveReplyLater(mid,
            [self = shared_from_this(), mid, hook = std::move(hook)]
            (error_code ec, Revision revision) {
                if(!ec) {
                    self->logOnRemoveSticker(mid, macs::StickerType::reply_later);
                }
                hook(std::move(ec), std::move(revision));
            });
}

void StickersRepository::logOnRemoveSticker(const Mid& mid, macs::StickerType stickerType) const {
    logOperation<params::RemoveSticker>(
                    id::state(mid),
                    id::affected(1ul),
                    id::mid(mid),
                    id::stickerSymbol(yamail::data::reflection::to_string(stickerType)));
}

} // namespace macs
