#include <macs/tabs_map.h>
#include <boost/range/algorithm.hpp>

namespace macs {
namespace deprecated {

const std::vector<std::string> TabsMap::defaultNames_ = {
    "default",
    "relevant"
};

Tab TabsMap::getTab(const LabelSet& labels, const std::string& tabName) const {
    Tab res;

    if (!isDefaultTab(tabName)) {
        const auto t = map_.find(tabName);
        if (t == map_.end()) {
            res.type = Tab::Type::unknown;
            return res;
        }
        const auto l = labels.find(t->second, Label::Type::spamDefense);
        if (l == labels.end()) {
            res.type = Tab::Type::empty;
            return res;
        }
        res.type = Tab::Type::withLabel;
        res.label = l->second;
    } else {
        for (const std::string& type: map_ | boost::adaptors::map_values) {
            const auto l = labels.find(type, Label::Type::spamDefense);
            if (l != labels.end()) {
                res.excludeLabels.push_back(l->second);
            }
        }
        res.type = (res.excludeLabels.empty()) ? Tab::Type::whole : Tab::Type::withoutLabels;
    }

    return res;
}

std::unordered_map<std::string, Tab> TabsMap::getAllTabs(const LabelSet& labels) const {
    std::unordered_map<std::string, Tab> res;

    boost::transform(map_, std::inserter(res, res.end()), [&](auto& t) {
        return std::make_pair(t.first, this->getTab(labels, t.first));
    });
    res.emplace(defaultTab(), getTab(labels, defaultTab()));

    return res;
}

std::pair<TabsMap::LabelTypes, TabsMap::LabelTypes>
        TabsMap::getLabelsToChange(const std::string& tabName) const {
    using namespace boost::adaptors;

    std::string newLabelType = getType(tabName);
    auto oldLabelTypes = map_ | map_values | filtered(
                [&newLabelType](auto& type) { return type != newLabelType; });

    return std::make_pair(
        (newLabelType.empty()) ? LabelTypes{} : LabelTypes{newLabelType},
        LabelTypes{std::begin(oldLabelTypes), std::end(oldLabelTypes)}
    );
}

std::string TabsMap::getType(const std::string& tabName) const {
    if (isDefaultTab(tabName)) {
        return "";
    }

    auto t = map_.find(tabName);
    if (t == map_.end()) {
        throw UnknownTabException("Unknown tab: " + tabName);
    }
    return t->second;
}

const std::string& TabsMap::defaultTab() {
    return defaultNames_.front();
}

bool TabsMap::isDefaultTab(const std::string& tabName) {
    return boost::count(defaultNames_, tabName);
}

TabsMap initTabsMap(const boost::property_tree::ptree& node, const std::string& nodeName) {
    TabsMap::Map map;
    auto confTabs = node.equal_range(nodeName);
    for (auto it = confTabs.first; it != confTabs.second; ++it) {
        const boost::property_tree::ptree& tab = it->second;
        map.emplace(tab.get<std::string>("name"), tab.get<std::string>("label_type"));
    }
    return {map};
}

} // namespace deprecated
} // namespace macs
