#include <internal/hooks/wrap.h>
#include <internal/query/ids.h>
#include <internal/reflection/backup.h>
#include <internal/backup/query.h>
#include <macs/backup_reflection.h>
#include <macs/backup_error.h>
#include <iterator>


namespace macs::pg {
namespace ydr = yamail::data::reflection;
namespace {
inline auto fromReflection(boost::optional<reflection::Backup>& data) {
    return data ? std::make_optional(Backup {
        .backup_id=std::move(data->backup_id),
        .created=std::move(data->created),
        .updated=std::move(data->updated),
        .message_count=std::move(data->message_count),
        .state=ydr::from_string<BackupState>(data->state),
        .notice=data->notice ? std::make_optional(*data->notice) : std::nullopt,
        .version=std::move(data->version)
    }) : std::nullopt;
}

inline auto fromReflection(const std::vector<reflection::FidToFid>& data) {
    std::map<macs::Fid, macs::Fid> ret;

    for (const auto& element: data) {
        ret[std::to_string(element.original)] = std::to_string(element.renewed);
    }

    return ret;
}

inline auto fromReflection(reflection::BackupSettings& data) {
    BackupSettings ret;

    for (const auto& element: data.fids) {
        ret.fids.push_back(std::to_string(element));
    }
    for (const auto& element: data.tabs) {
        ret.tabs.push_back(macs::Tab::Type::fromString(element));
    }

    return ret;
}

inline auto fromReflection(boost::optional<reflection::Restore>& data) {
    return data ? std::make_optional(Restore {
        .backup_id=std::move(data->backup_id),
        .created=std::move(data->created),
        .updated=std::move(data->updated),
        .to_restore_count=std::move(data->to_restore_count),
        .restored_count=std::move(data->restored_count),
        .state=ydr::from_string<RestoreState>(data->state),
        .method=ydr::from_string<RestoreMethod>(data->method),
        .notice=data->notice ? std::make_optional(*data->notice) : std::nullopt,
        .fids_mapping=fromReflection(data->fids_mapping)
    }) : std::nullopt;
}

inline auto fromReflection(reflection::BackupStatus& data) {
    return BackupStatus {
        .primary = fromReflection(data.primary),
        .additional = fromReflection(data.additional),
        .restore = fromReflection(data.restore)
    };
}

inline auto fromReflection(boost::optional<std::vector<int32_t>>& data) {
    std::vector<macs::Fid> ret;
    if (data) {
        ret.reserve(data->size());
        for (auto i: *data) {
            ret.push_back(std::to_string(i));
        }
    }
    return ret;
}

inline auto fromReflection(reflection::BackupFolder& data) {
    return macs::BackupFolder {
        .fid=std::to_string(data.fid),
        .name=std::move(data.name),
        .parentFid=data.parent_fid ? std::make_optional(std::to_string(*data.parent_fid))
                                   : std::nullopt,
        .isBackuped=data.is_backuped
    };
}

inline auto fromReflection(reflection::BackupId& data) {
    return data.backup_id;
}

inline auto fromReflection(reflection::MessagesToRestore& data) {
    return BackupMessage {
        .mid=std::to_string(data.mid),
        .st_id=data.st_id,
        .fid=std::to_string(data.fid),
        .tab=(data.tab ? std::make_optional(Tab::Type::fromString(data.tab.get(), std::nothrow)) : std::nullopt),
        .receivedDate=data.received_date,
        .attributes=data.attributes
    };
}

template<class Result>
decltype(auto) onResult(OnExecute handler) {
    using OptStr = boost::optional<std::string>;
    Hook<OptStr> hook = [h = std::move(handler)] (error_code ec, const auto& result) {
        if (ec) {
            h(ec);
        } else if (result) {
            h(make_error(ydr::from_string<Result>(*result)));
        } else {
            h(error_code());
        }
    };

    return wrapHook<reflection::WithResult>(
        std::move(hook),
        [] (reflection::WithResult data) {
            return data.result;
        }
    );
}

}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncStatus(OnBackupStatus handler) const {
    db()->fetch(
        query<query::BackupStatus>(),
        wrapHook<reflection::BackupStatus>(
            std::move(handler),
            [] (reflection::BackupStatus data) {
                return fromReflection(data);
            }
        )
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncSettings(OnBackupSettings handler) const {
    db()->fetch(
        query<query::BackupSettings>(),
        wrapHook<reflection::BackupSettings>(
            std::move(handler),
            [] (reflection::BackupSettings data) {
                return fromReflection(data);
            }
        )
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncChangeSettings(const BackupSettings& settings, OnBackupSettings handler) const {
    db()->fetch(
        queryUpdate<query::BackupChangeSettings>(settings.fids, settings.tabs),
        wrapHook<reflection::BackupSettings>(
            std::move(handler),
            [] (reflection::BackupSettings data) {
                return fromReflection(data);
            }
        )
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncCreateBackup(macs::BackupId id, int32_t maxMessages, bool useTabs, OnExecute handler) const {
    db()->fetch(
        queryUpdate<query::BackupCreate>(
            query::BackupId(id), query::MaxMessagesInBackup(maxMessages), query::UseTabs(useTabs)
        ),
        onResult<CreateBackupStatus>(std::move(handler))
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncFillBackup(macs::BackupId id, bool useTabs, OnExecute handler) const {
    db()->fetch(
        queryUpdate<query::BackupFill>(query::BackupId(id), query::UseTabs(useTabs)),
        onResult<FillBackupStatus>(std::move(handler))
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncId(OnBackupId handler) const {
    db()->fetch(
        queryUpdate<query::BackupNextId>(),
        wrapHook<reflection::BackupId>(
            std::move(handler),
            [] (reflection::BackupId data) {
                return fromReflection(data);
            }
        )
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncRestore(macs::BackupId id, std::time_t now,
                                                        macs::RestoreMethod method,
                                                        macs::BackupFidsMapping mapping,
                                                        OnExecute handler) const {
    macs::BackupFidToFidVec fidToFid;
    for (const auto& val: mapping) {
        fidToFid.push_back(std::make_pair(std::stoi(val.first), std::stoi(val.second)));
    }

    db()->fetch(
        queryUpdate<query::BackupRestore>(
            query::BackupId(id), query::CreationTs(now),
            query::RestoreMethod(std::string(ydr::to_string(std::move(method)))),
            query::BackupFidToFidVec(std::move(fidToFid))
        ),
        onResult<RestoreBackupStatus>(std::move(handler))
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncFolders(macs::BackupId id, OnBackupFolders handler) const {
    db()->fetch(
        query<query::BackupFolders>(query::BackupId(id)),
        wrapHook<reflection::BackupFolder>(
            std::move(handler),
            [] (reflection::BackupFolder data) {
                return fromReflection(data);
            }
        )
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncSelectMessages(macs::BackupId id, uint64_t count, OnBackupBox handler) const {
    db()->fetch(
        query<query::BackupMessagesToRestore>(query::BackupId(id), query::RowCount(count)),
        wrapHook<reflection::MessagesToRestore>(
            std::move(handler),
            [] (reflection::MessagesToRestore data) {
                return fromReflection(data);
            }
        )
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncSelectMessagesInFolder(macs::BackupId id, uint64_t count, macs::Fid fid, OnBackupBox handler) const {
    db()->fetch(
            query<query::BackupMessagesInFolderToRestore>(query::BackupId(id), query::RowCount(count), query::FolderId(fid)),
            wrapHook<reflection::MessagesToRestore>(
                    std::move(handler),
                    [] (reflection::MessagesToRestore data) {
                        return fromReflection(data);
                    }
            )
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncDeleteMessages(macs::BackupId id, macs::MidVec mids, OnExecute handler) const {
    db()->execute(
        query<query::BackupDeleteMessages>(
            query::BackupId(id),
            query::MailIds(std::move(mids))
        ),
        std::move(handler)
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncUpdateMessages(macs::BackupId id, macs::BackupMidToMid mapping, OnExecute handler) const {
    macs::BackupMidToMidVec vec;
    for (const auto& val: mapping) {
        vec.push_back(std::make_pair(std::stoll(val.first), std::stoll(val.second)));
    }

    db()->execute(
        queryUpdate<query::BackupUpdateMids>(query::BackupId(id), query::BackupMidToMidVec(vec)),
        std::move(handler)
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncCompleteRunningRestore(OnExecute handler) const {
    db()->execute(
        queryUpdate<query::BackupRestoreComplete>(),
        std::move(handler)
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncFailRunningRestore(std::string notice, OnExecute handler) const {
    db()->execute(
        queryUpdate<query::BackupRestoreFail>(query::FailReason(std::move(notice))),
        std::move(handler)
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncFailRunningBackup(std::string notice, OnExecute handler) const {
    db()->execute(
        queryUpdate<query::BackupFail>(query::FailReason(std::move(notice))),
        std::move(handler)
    );
}

template <typename DatabaseGenerator>
void BackupsRepository<DatabaseGenerator>::asyncDeactivateBackup(OnExecute handler) const {
    db()->fetch(
        queryUpdate<query::BackupDeactivate>(),
        onResult<DeactivateBackupStatus>(std::move(handler))
    );
}

}
