#pragma once

#include <macs_pg/changelog/change_type.h>
#include <macs_pg/changelog/repository.h>
#include <macs/hooks.h>

#include <pgg/database/database.h>
#include <pgg/query/repository.h>
#include <internal/query/comment.h>

namespace macs {
namespace pg {

template <typename DatabaseGenerator>
class ChangeLogRepository : public macs::ChangeLogRepository {
public:
    typedef macs::ChangeLogRepository Base;

    ChangeLogRepository(const DatabaseGenerator& generator,
                        pgg::query::RepositoryPtr queryRepository,
                        pgg::Milliseconds /*transactionTimeout*/)
    : Base(), queryRepository_(queryRepository), db(std::move(generator))
    {}

    virtual ~ChangeLogRepository() = default;

private:
    // macs methods
    void asyncGetChanges(std::vector<ChangeId> ids,
                         OnChangesReceive hook) const override;

    void asyncGetMimimumKnownChangeId(OnChangeIdReceive hook) const override;

    const pgg::query::Repository& queryRepository() const {
        return *queryRepository_;
    }
    std::shared_ptr<const ChangeLogRepository> getSelf() const {
        return std::dynamic_pointer_cast<const ChangeLogRepository>(pgg::share(this));
    }

    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
};

template <typename DatabaseGenerator>
class UserChangeLogRepository : public macs::UserChangeLogRepository {
public:

    UserChangeLogRepository(Uid uid,
                        DatabaseGenerator generator,
                        pgg::query::RepositoryPtr queryRepository,
                        pgg::Milliseconds /*transactionTimeout*/)
    : uid_(uid), queryRepository_(queryRepository), db(std::move(generator))
    {}

private:
    // macs methods
    void asyncGetChanges(Revision start, std::int64_t limit,
                         OnChangesReceive hook) const override;

    void asyncGetChangelog(Revision lastRevision, std::int64_t limit,
                           OnChangesReceive hook) const override;

    void asyncGetChangelogByType(Revision lastRevision, std::int64_t limit,
                                 const std::vector<macs::pg::ChangeType>& changeTypes,
                                 OnChangesReceive hook) const override;

    void asyncGetChangeIdByRevision(Revision revision, OnChangesIdReceive hook) const override;

    const pgg::query::Repository& queryRepository() const {
        return *queryRepository_;
    }
    std::shared_ptr<const UserChangeLogRepository> getSelf() const {
        return std::dynamic_pointer_cast<const UserChangeLogRepository>(pgg::share(this));
    }

    const Uid& uid() const { return uid_;}

    template <typename T, typename ...ArgsT >
    T makeQuery(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                std::forward<ArgsT>(args) ...);
    }

    Uid uid_;
    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
};

template <typename DatabaseGenerator>
UserChangeLogRepositoryPtr createUserChangeLogRepository(
    const std::string& uid,
    DatabaseGenerator dbg,
    pgg::query::RepositoryPtr repo,
    pgg::Milliseconds timeout);

} // namespace pg
} // namespace macs

#include "repository.ipp"
