#include <utility>

#include <macs_pg/changelog/change_type.h>
#include <macs_pg/changelog/factory.h>
#include <internal/changelog/repository.h>
#include <internal/changelog/query.h>

#include <pgg/cast.h>
#include <internal/hooks/wrap.h>
#include <internal/reflection/change.h>

namespace macs::pg {

namespace detail {
    inline macs::Change changeFromReflection(const reflection::Change & v) {
        return macs::ChangeFactory()
            .changeId(v.cid)
            .revision(PGG_NUMERIC_CAST(Revision, v.revision))
            .type(ChangeType::fromString(v.type, std::nothrow))
            .changed(v.changed)
            .arguments(v.arguments)
            .uid(std::to_string(v.uid))
            .release();
    }
}


template <typename DatabaseGenerator>
void ChangeLogRepository<DatabaseGenerator>::asyncGetChanges(std::vector<ChangeId> ids,
                                         OnChangesReceive hook) const {
    const auto q = queryRepository().template query<query::ChangesById>(ids);
    db()->fetch(q, wrapHook<reflection::Change>(std::move(hook), detail::changeFromReflection));
}

template <typename DatabaseGenerator>
void ChangeLogRepository<DatabaseGenerator>::asyncGetMimimumKnownChangeId(OnChangeIdReceive hook) const {
    const auto q = queryRepository().template query<query::MinimumKnownChangeId>();
    db()->fetch(q, wrapHook<reflection::ChangeId>(std::move(hook),
            [] (auto r) { return r.cid; }));
}

template <typename DatabaseGenerator>
void UserChangeLogRepository<DatabaseGenerator>::asyncGetChanges(Revision start, std::int64_t limit,
                                         OnChangesReceive hook) const {
    const auto q = makeQuery<query::ChangesByUid>(
                                            query::UserId(uid()),
                                            query::Revision(start),
                                            query::Limit(limit));
    db()->fetch(q, wrapHook<reflection::Change>(std::move(hook), detail::changeFromReflection));
}

template <typename DatabaseGenerator>
void UserChangeLogRepository<DatabaseGenerator>::asyncGetChangeIdByRevision(Revision revision,
                                                         OnChangesIdReceive hook) const {
    const auto q = makeQuery<query::GetChangeIdByRevision>(
                                            query::UserId(uid()),
                                            query::Revision(revision));
    using R = reflection::ChangeId;
    db()->fetch(q, wrapHook<R>(std::move(hook), [] (R r) { return r.cid; }));
}

template <typename DatabaseGenerator>
void UserChangeLogRepository<DatabaseGenerator>::asyncGetChangelog(Revision lastRevision, std::int64_t limit,
                                                OnChangesReceive hook) const {
    const auto q = makeQuery<query::ChangelogByUid>(
                                            query::UserId(uid()),
                                            query::Revision(lastRevision),
                                            query::Limit(limit));
    db()->fetch(q, wrapHook<reflection::Change>(std::move(hook), detail::changeFromReflection));
}

template <typename DatabaseGenerator>
void UserChangeLogRepository<DatabaseGenerator>::asyncGetChangelogByType(Revision lastRevision, std::int64_t limit,
                                                      const std::vector<macs::pg::ChangeType>& changeTypes,
                                                      OnChangesReceive hook) const {
    const auto q = makeQuery<query::ChangelogByUidAndType>(
                                            query::UserId(uid()),
                                            query::Revision(lastRevision),
                                            query::Limit(limit),
                                            query::ChangeLogTypes(changeTypes));
    db()->fetch(q, wrapHook<reflection::Change>(std::move(hook), detail::changeFromReflection));
}

} // namespace macs::pg
