#pragma once

#include <macs_pg/collectors/collector_factory.h>
#include <macs_pg/collectors/hooks.h>
#include <internal/collectors/collector_id_result.h>
#include <internal/collectors/query.h>
#include <internal/collectors/query_helpers.h>

#include <pgg/query/transactional.h>
#include <pgg/cast.h>
#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

class MigrateCollector {
public:
    MigrateCollector(pgg::query::RepositoryPtr qr, OnCollectorResult hook,
            const Uid& uid, CollectorFactory factory, pgg::Milliseconds timeout)
        : queryRepository(qr),
          hook(std::move(hook)),
          uid(uid),
          factory(std::move(factory)),
          transactionTimeout(timeout)
    {
    }

    template <typename Transactional>
    void operator()(pgg::query::Coroutine & coro, Transactional & t) {
        using namespace macs::pg::query;
        auto onError = boost::bind(&MigrateCollector::checkError, this, _1);
        reenter( coro ) {
            yield t.begin(onError, transactionTimeout);
            yield {
                auto q = makeQueryCreateCollector(*queryRepository, uid, factory);
                t.fetch(q, [&](error_code err, auto range) {
                    onCreate(std::move(err), std::move(range));
                });
            }

            if (error) {
                return hook(error, CollectorResult());
            }

            yield {
                auto q = makeQueryAddMigrationData(*queryRepository, uid, factory);
                t.execute(q, onError);
            }
            yield t.commit(onError);

            factory.state(macs::CollectorState::migrated);
            factory.migratedLastMid(factory.product().lastMid());
            hook(error_code(), CollectorResult{revision, factory.release()});
        }
    }

private:
    void checkError(error_code err) {
        if (err) {
            hook(err, CollectorResult());
        }
    }

    template <typename DataRange>
    void onCreate(error_code err, DataRange range) {
        if (err) {
            hook(err);
        } else if (!range.empty()) {
            auto v = pgg::cast<reflection::CollectorIdResult>(range.front());
            revision = PGG_NUMERIC_CAST(Revision, v.revision);
            factory
                .uid(uid)
                .collectorId(CollectorId(v.collector_id));
        } else {
            error = error_code(pgg::error::noDataReceived,
                "No data received as a result of CreateCollector");
        }
    }

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return queryRepository->query<T>(query::UserId(uid), std::forward<Args>(args)...);
    }

    pgg::query::RepositoryPtr queryRepository;
    OnCollectorResult hook;
    Uid uid;
    CollectorFactory factory;
    pgg::Milliseconds transactionTimeout;
    error_code error;

    Revision revision;
};

} // namespace pg
} // namespace macs

#include <boost/asio/unyield.hpp>
