#pragma once

#include <macs_pg/collectors/collector_factory.h>
#include <internal/collectors/query.h>
#include <internal/query/comment.h>

namespace macs::pg {

inline auto makeQueryCreateCollector(const pgg::query::Repository& qr, const Uid& uid, const CollectorFactory& factory) {
    auto& product = factory.product();
    if (product.srcUid().empty()) {
        throw std::runtime_error("can't create collector with empty srcUid");
    }
    auto srcUid = query::SrcUserId(product.srcUid());
    auto authToken = query::CollectorsAuthToken(product.authToken());
    auto rootFid = query::OptionalFolderId();
    if (!product.rootFolderId().empty()) {
        rootFid = query::OptionalFolderId(product.rootFolderId());
    }
    auto lid = query::OptionalLabelId();
    if (!product.labelId().empty()) {
        lid = query::OptionalLabelId(product.labelId());
    }

    return makeQueryWithComment<query::CreateCollector>(qr, uid, srcUid, authToken, rootFid, lid);
}

inline auto makeQueryAddMigrationData(const pgg::query::Repository& qr, const Uid& uid, const CollectorFactory& factory) {
    auto& product = factory.product();
    if (!product.collectorId()) {
        throw std::runtime_error("can't add migrationd data without collectorId");
    }
    auto collectorId = query::CollectorId(product.collectorId());
    if (product.lastMid().empty()) {
        throw std::runtime_error("can't add migrationd data with empty lastMid");
    }
    auto ignoreFoldersStruct = query::CollectorsIgnoreFoldersStruct(product.ignoreFoldersStruct());
    auto lastMid = query::CollectorsLastMid(product.lastMid());
    auto skippedMids = query::CollectorsSkippedMids(product.skippedMids());
    if (!product.oldPopid()) {
        throw std::runtime_error("can't add migrationd data without oldPopid");
    }
    auto oldPopid = query::OldPopid(product.oldPopid());
    if (!product.creationTs()) {
        throw std::runtime_error("can't add migrationd data without creationTs");
    }
    auto creationTs = query::CreationTs(product.creationTs());
    if (product.originalServer().empty()) {
        throw std::runtime_error("can't add migrationd data without originalServer");
    }
    auto originalServer = query::CollectorsOriginalServer(product.originalServer());
    if (product.migrationTargetState() == macs::CollectorState::undefined) {
        throw std::runtime_error("can't add migrationd data without migrationTargetState");
    }
    auto migrationTargetState = query::CollectorsState(product.migrationTargetState());
    return makeQueryWithComment<query::AddMigrationData>(qr,
                                                         uid,
                                                         collectorId,
                                                         ignoreFoldersStruct,
                                                         lastMid,
                                                         skippedMids,
                                                         oldPopid,
                                                         creationTs,
                                                         originalServer,
                                                         migrationTargetState);
}

inline auto makeQueryResetCollector(const pgg::query::Repository& qr, const Uid& uid, const CollectorFactory& factory) {
    auto& product = factory.product();
    if (!product.collectorId()) {
        throw std::invalid_argument("can't reset collector without collectorId");
    }
    auto collectorId = query::CollectorId(product.collectorId());
    auto state = query::CollectorsState(product.state());
    if (product.authToken().empty()) {
        throw std::invalid_argument("can't reset collector with empty authToken");
    }
    auto authToken = query::CollectorsAuthToken(product.authToken());
    if (product.rootFolderId().empty()) {
        throw std::invalid_argument("can't reset collector with empty rootFolderId");
    }
    auto rootFid = query::FolderId(product.rootFolderId());
    if (product.labelId().empty()) {
        throw std::invalid_argument("can't reset collector with empty labelId");
    }
    auto lid = query::LabelId(product.labelId());
    if (product.lastMid().empty()) {
        throw std::invalid_argument("can't reset collector with empty lastMid");
    }
    auto lastMid = query::CollectorsLastMid(product.lastMid());
    auto skippedMids = query::CollectorsSkippedMids(product.skippedMids());
    if (!product.creationTs()) {
        throw std::invalid_argument("can't reset collector without creationTs");
    }
    auto creationTs = query::CreationTs(product.creationTs());

    return makeQueryWithComment<query::ResetCollector>(
        qr, uid, collectorId, state, authToken, rootFid, lid, lastMid, skippedMids, creationTs);
}

}
