#include <internal/collectors/collector_id_result.h>
#include <internal/collectors/migrate_collector.h>
#include <internal/collectors/query.h>
#include <internal/collectors/query_helpers.h>
#include <internal/collectors/repository.h>
#include <internal/envelope/factory.h>
#include <internal/hooks/wrap.h>
#include <internal/query/ids.h>

#include <pgg/numeric_cast.h>

namespace macs::pg {

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncCreateCollector(
        CollectorFactory factory, OnCollectorResult hook) const {
    auto factoryPtr = std::make_shared<CollectorFactory>(std::move(factory));
    factoryPtr->uid(uid());
    auto q = makeQueryCreateCollector(queryRepository(), uid(), *factoryPtr);
    auto converter = [factoryPtr](const macs::pg::reflection::CollectorIdResult& v) {
            factoryPtr->collectorId(CollectorId(v.collector_id))
                .creationTs(std::time(nullptr))
                .state(macs::CollectorState::created);
            return CollectorResult{PGG_NUMERIC_CAST(Revision, v.revision), factoryPtr->release()};
        };
    auto wrapper = wrapHook<reflection::CollectorIdResult>(std::move(hook), std::move(converter));
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncMigrateCollector(
        CollectorFactory factory, OnCollectorResult hook) const {
    auto migrateCollector = MigrateCollector(queryRepository_, std::move(hook), uid(), std::move(factory), transactionTimeout_);
    pgg::query::fallback::runTransactional(db, queryRepository_, std::move(migrateCollector));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncResetCollector(const CollectorFactory& factory, OnUpdate hook) const {
    auto q = makeQueryResetCollector(queryRepository(), uid(), factory);
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncDeleteCollector(CollectorId collectorId, OnUpdate hook) const {
    const auto q = makeQuery<query::DeleteCollector>(query::CollectorId(collectorId));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncUpdateSkippedMids(CollectorId collectorId,
        const MidVec &skippedMids, OnExecute hook) const {
    const auto q = makeQuery<query::UpdateCollectorsSkippedMids>(query::CollectorId(collectorId),
                                                                 query::CollectorsSkippedMids(skippedMids));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncUpdateLastMid(CollectorId collectorId,
        const Mid &lastMid, OnExecute hook) const {
    const auto q = makeQuery<query::UpdateCollectorsLastMid>(query::CollectorId(collectorId),
                                                             query::CollectorsLastMid(lastMid));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncUpdateState(CollectorId collectorId,
        const macs::CollectorState &newState, OnUpdate hook) const {
    const auto q = makeQuery<query::UpdateCollectorsState>(query::CollectorId(collectorId),
                                                           query::CollectorsState(newState));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncUpdateMigrationTargetState(CollectorId collectorId,
        const macs::CollectorState &newState, OnUpdate hook) const {
    const auto q = makeQuery<query::UpdateCollectorsMigrationTargetState>(query::CollectorId(collectorId),
                                                                          query::CollectorsState(newState));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncUpdateCollector(const CollectorFactory& factory, OnUpdate hook) const {
    auto& product = factory.product();

    if (!product.collectorId()) {
        throw std::runtime_error("can't edit collector with empty collectorId");
    }
    auto collectorId = query::CollectorId(product.collectorId());
    if (product.authToken().empty()) {
        throw std::runtime_error("can't set authToken to empty, use resetToken instead");
    }
    auto authToken = query::CollectorsAuthToken(product.authToken());
    if (product.rootFolderId().empty()) {
        throw std::runtime_error("can't set rootFolderId to empty");
    }
    auto rootFid = query::FolderId(product.rootFolderId());
    if (product.labelId().empty()) {
        throw std::runtime_error("can't set labelId to empty");
    }
    auto lid = query::LabelId(product.labelId());

    const auto q = makeQuery<query::EditCollector>(collectorId, authToken, rootFid, lid);
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncResetToken(CollectorId collectorId, OnExecute hook) const {
    const auto q = makeQuery<query::ResetCollectorsToken>(query::CollectorId(collectorId));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncGetNextEnvelopeChunk(const Mid &id, const uint64_t count,
        LabelSet labels, OnEnvelopeChunkReceive hook) const {
    auto q = makeQuery<query::MailboxEntriesAfterMid>(query::MailId(id), query::RowCount(count));
    auto handler = wrapHook<reflection::Envelope>(std::move(hook),
        [labels = std::move(labels)](reflection::Envelope envelope) {
            return makeEnvelope(labels, std::move(envelope));
        });
    db()->fetch(q, handler);
}

template <typename DatabaseGenerator>
void CollectorsRepository<DatabaseGenerator>::asyncGetNextPop3EnvelopeChunk(const Mid &id, const uint64_t count,
        LabelSet labels, OnEnvelopeChunkReceive hook) const {
    auto q = makeQuery<query::MailboxEntriesAfterMidInPop3Box>(query::MailId(id), query::RowCount(count));
    auto handler = wrapHook<reflection::Envelope>(std::move(hook),
        [labels = std::move(labels)](reflection::Envelope envelope) {
            return makeEnvelope(labels, std::move(envelope));
        });
    db()->fetch(q, handler);
}

}
