#pragma once

#include <pgg/query/boundaries.h>
#include <pgg/query/ranges.h>
#include <internal/query/ids.h>
#include <internal/envelope/message_attributes.h>
#include <internal/envelope/message_attach.h>
#include <internal/envelope/recipient.h>
#include <boost/hana/for_each.hpp>
#include <boost/hana/tuple.hpp>
#include <boost/hana/transform.hpp>

namespace macs {
namespace pg {
namespace query {

class FlagsAttributesAndLids {
public:
    using Sym = Label::Symbol;

    class MessageFlags {
        bool seen_ = false;
        bool deleted_ = false;
    public:
        bool seen() const { return seen_;}
        bool deleted() const { return deleted_;}

        bool handle( const Sym & sym ) {
            if (sym == Sym::seen_label) {
                seen_ = true;
            } else if (sym == Sym::deleted_label) {
                deleted_ = true;
            } else {
                return false;
            }
            return true;
        }
    };

    class MessageAttributesList {
    public:
        using Value = std::vector<MessageAttributes>;

        bool handle( const Sym & sym ) {
            auto attr = MessageAttrubutesConverter().attr(sym);
            if( attr == MessageAttributes::unknown ) {
                return false;
            }
            value_.push_back(attr);
            return true;
        }

        const Value & value() const { return value_;}
    private:
        Value value_;
    };

    class RealLabelsIdList {
    public:
        using Value = std::vector<std::string>;
        void handle( const std::string & lid ) {
            value_.push_back(lid);
        }

        const Value & value() const { return value_;}
    private:
        Value value_;
    };

    const MessageFlags & flags() const { return flags_; }
    const MessageAttributesList & attrs() const { return attrs_; }
    const RealLabelsIdList & lids() const { return lids_; }
public:
    template <typename Range>
    FlagsAttributesAndLids(Range lids) {
        for(const auto & l : lids) {
            const auto & labels = fakeLabels();
            const auto i = labels.find(l);
            if(i != labels.end()) {
                const auto & sym = i->symbolicName();
                if (!skipLabel(sym) && !flags_.handle(sym) && !attrs_.handle(sym)) {
                    std::ostringstream s;
                    s << __PRETTY_FUNCTION__ << ": unexpected lid " << l
                            << " which is not a flag or attribute or label "
                            << std::endl;
                    throw std::logic_error(s.str());
                }
            } else {
                lids_.handle(l);
            }
        }
    }
    FlagsAttributesAndLids() = default;
private:
    bool skipLabel( const Sym & sym ) const {
        // we do not control "recent" via StoreMessageor QuickSaveMessage
        return sym == Sym::attached_label || sym == Sym::recent_label;
    }
    MessageFlags flags_;
    MessageAttributesList attrs_;
    RealLabelsIdList lids_;
};

template<typename macsEnvelope>
class EnvelopeTemplate {
protected:
    auto makeAttaches() const {
        using MA = MessageAttach;
        using StoreAttach = boost::tuple<MA::Hid, MA::ContentType,
                                    MA::Filename, MA::Size>;
        std::vector<StoreAttach> att;
        boost::transform(v_.attachments(), std::back_inserter(att),
                [](AttachmentDescriptor a) {
            return StoreAttach( std::move(a.m_hid), std::move(a.m_contentType),
                    std::move(a.m_fileName), a.m_size);
        });
        return att;
    }

    auto makeRecipients() const {
        using RT = Recipient::Type;
        std::vector<Recipient> recipients;

        auto fillRecipients = [&recipients]
            (const std::string& emailsString,
            const macs::EmailsOpt& emailsParsed,
            Recipient::Type type) mutable
            {
                if (!emailsParsed) {
                    fromStringWithType(emailsString, type, recipients);
                } else {
                    for (const auto& email : *emailsParsed) {
                        recipients.emplace_back(email.displayName(), email.addressString(), type);
                    }
                }
            };

        fillRecipients(v_.from(), v_.parsedFrom(), RT::from);
        fillRecipients(v_.to(), v_.parsedTo(), RT::to);
        fillRecipients(v_.replyTo(), v_.parsedReplyTo(), RT::replyTo);
        fillRecipients(v_.cc(), v_.parsedCc(), RT::cc);
        fillRecipients(v_.bcc(), v_.parsedBcc(), RT::bcc);
        fillRecipients(v_.sender(), v_.parsedSender(), RT::sender);

        std::vector<Recipient::Tuple> recipientTuples;
        for(auto & r : recipients) {
            recipientTuples.emplace_back(std::move(r.tuple()));
        }
        return recipientTuples;
    }

    auto makeThreadId() const {
        const auto & t = v_.threadId();
        boost::optional<std::string> tid;
        if( !t.empty() && t!="0" ) {
            tid = t;
        }
        return tid;
    }

    auto makeTab() const {
        const auto& t = v_.tab();
        boost::optional<std::string> tab;
        if (t) {
            tab = boost::make_optional<std::string>(t.value().toString());
        }
        return tab;
    }

public:
    EnvelopeTemplate() {}
    explicit EnvelopeTemplate(macsEnvelope v)
    : v_(std::move(v)), flagsAttributesAndLids_(macs().labels()){}

    const FlagsAttributesAndLids & flagsAttributesAndLids() const {
        return flagsAttributesAndLids_;
    }

    const macsEnvelope& macs() const { return v_; }

protected:
    macsEnvelope v_;
    FlagsAttributesAndLids flagsAttributesAndLids_;
};

class Envelope : public EnvelopeTemplate<macs::Envelope> {
public:
    Envelope() : EnvelopeTemplate() {}
    explicit Envelope(macs::Envelope v) : EnvelopeTemplate<macs::Envelope>(v) {}

    auto makeFields() const {
        return boost::hana::make_tuple(
            boost::hana::make_pair(flagsAttributesAndLids_.flags().seen(), "seen"),
            boost::hana::make_pair(flagsAttributesAndLids_.flags().deleted(), "deleted"),
            boost::hana::make_pair(flagsAttributesAndLids_.attrs().value(), "attributes"),
            boost::hana::make_pair(flagsAttributesAndLids_.lids().value(), "lids"),
            boost::hana::make_pair(makeAttaches(), "attaches"),
            boost::hana::make_pair(makeRecipients(), "recipients"),
            boost::hana::make_pair(v_.mid(), "mid"),
            boost::hana::make_pair(v_.fid(), "fid"),
            boost::hana::make_pair(makeThreadId(), "tid"),
            boost::hana::make_pair(v_.date(), "hdr_date"),
            boost::hana::make_pair(v_.receiveDate(), "received_date"),
            boost::hana::make_pair(v_.subject(), "subject"),
            boost::hana::make_pair(v_.stid(), "st_id"),
            boost::hana::make_pair(v_.firstline(), "firstline"),
            boost::hana::make_pair(v_.rfcId(), "hdr_message_id"),
            boost::hana::make_pair(v_.size(), "size"),
            boost::hana::make_pair(v_.extraData(), "extra_data"),
            boost::hana::make_pair(makeTab(), "tab_type")
        );
    }

    template <typename Mapper>
    void map( const Mapper & m ) const {
        boost::hana::for_each(makeFields(), [&](auto&& field) {
            m.mapValue(boost::hana::first(field), boost::hana::second(field));
        });
    }

    auto get() const {
        return boost::hana::transform(makeFields(), [](auto&& field) {
            return boost::hana::first(field);
        });
    }
};

class EnvelopeDeleted : public EnvelopeTemplate<macs::EnvelopeDeleted> {
public:
    EnvelopeDeleted() : EnvelopeTemplate() {}
    explicit EnvelopeDeleted(macs::EnvelopeDeleted v) : EnvelopeTemplate<macs::EnvelopeDeleted>(v) {}

    auto makeFields() const {
        return boost::hana::make_tuple(
            boost::hana::make_pair(flagsAttributesAndLids_.attrs().value(), "attributes"),
            boost::hana::make_pair(makeAttaches(), "attaches"),
            boost::hana::make_pair(makeRecipients(), "recipients"),
            boost::hana::make_pair(v_.mid(), "mid"),
            boost::hana::make_pair(v_.date(), "hdr_date"),
            boost::hana::make_pair(v_.receiveDate(), "received_date"),
            boost::hana::make_pair(v_.subject(), "subject"),
            boost::hana::make_pair(v_.stid(), "st_id"),
            boost::hana::make_pair(v_.firstline(), "firstline"),
            boost::hana::make_pair(v_.rfcId(), "hdr_message_id"),
            boost::hana::make_pair(v_.size(), "size"),
            boost::hana::make_pair(v_.extraData(), "extra_data")
        );
    }

    template <typename Mapper>
    void map( const Mapper & m ) const {
        boost::hana::for_each(makeFields(), [&](auto&& field) {
            m.mapValue(boost::hana::first(field), boost::hana::second(field));
        });
    }

    auto get() const {
        return boost::hana::transform(makeFields(), [](auto&& field) {
            return boost::hana::first(field);
        });
    }
};

} // namespace query
} // namespace pg
} // namespace macs

namespace pgg {
namespace query {

template<typename Base, typename macsEnvelope, typename queryEnvelope>
struct EnvelopeHelper {
public:
    void set(macsEnvelope v) {
        queryEnvelope e(std::move(v));
        this->set(std::move(e));
    }
    Base& envelope(macsEnvelope v) {
        this->set(std::move(v));
        return static_cast<Base&>(*this);
    }

    void set(queryEnvelope v) { v_ = std::move(v); }
    Base& envelope(queryEnvelope v) {
        this->set(std::move(v));
        return static_cast<Base&>(*this);
    }

    template <typename MapperT>
    void map(const MapperT & m) const {
        v_.map(m);
    }

    auto get() const {
        return v_.get();
    }
private:
    queryEnvelope v_;
};

template<typename Base>
struct Helper<Base, macs::Envelope> : public EnvelopeHelper <Base, macs::Envelope, macs::pg::query::Envelope> {};

template<typename Base>
struct Helper<Base, macs::EnvelopeDeleted> : public EnvelopeHelper <Base, macs::EnvelopeDeleted, macs::pg::query::EnvelopeDeleted> {};

} // namespace query
} // namespace pgg
