#pragma once

#include <macs/data.h>
#include <macs/envelope_factory.h>
#include <pgg/numeric_cast.h>
#include <internal/envelope/message_attributes.h>
#include <internal/envelope/spam_defence_type.h>
#include <internal/envelope/message_attach.h>
#include <internal/envelope/recipient.h>
#include <internal/envelope/mime_parts.h>
#include <internal/reflection/envelope.h>
#include <string>

namespace macs {
namespace pg {

class EnvelopeFactory : public macs::EnvelopeFactory {
public:
    typedef macs::EnvelopeFactory Base;

    EnvelopeFactory(const LabelSet &labels) : labels(labels) {}

    void addAttribute(const MessageAttributes& a) {
        const std::string & lid = attributeToLid(a);
        if(!lid.empty()) {
            Base::addLabelID( lid );
        }
    }

    std::string append( const std::string & a, const std::string & b ) const {
        return a.empty() ? b : a + ',' + b;
    }

    void addRecipient(const Recipient::Reflection & t) {
        const Recipient r(t);
        const std::string value = emailString(r);
        using RT = Recipient::Type;
        switch( r.type() ) {
            case RT::from :
                Base::from(value);
                break;
            case RT::to :
                Base::to(append(Base::product().to(), value));
                break;
            case RT::replyTo :
                Base::replyTo(append(Base::product().replyTo(), value));
                break;
            case RT::cc :
                Base::cc(append(Base::product().cc(), value));
                break;
            case RT::bcc :
                Base::bcc(append(Base::product().bcc(), value));
                break;
            case RT::replyToAll :
                Base::replyTo(append(Base::product().replyTo(), value));
                break;
            case RT::sender :
                Base::sender(value);
                break;
            case RT::unknown :
                break;
        }
    }

    using Sym = Label::Symbol;
    void addAttach(MessageAttach::Reflection a) {
        MessageAttach ma(std::move(a));
        Base::addAttachment(toAttachmentDescriptor( ma ));
    }

    void setSeen( bool v ) {
        if( v ) {
            Base::addLabelID(getFakeLabelId(Sym::seen_label));
        }
    }

    void setRecent( bool v ) {
        if( v ) {
            Base::addLabelID(getFakeLabelId(Sym::recent_label));
        }
    }

    void setDeleted( bool v ) {
        if( v ) {
            Base::addLabelID(getFakeLabelId(Sym::deleted_label));
        }
    }

    Stid trimMulcaId(const std::string & id) const {
        static const std::string prefix = "mulca:2:";
        if (id.length() >= prefix.length() &&
            std::equal(prefix.begin(), prefix.end(), id.begin()))
        {
            return id.substr(prefix.length());
        } else {
            return id;
        }
    }

    void stId( const std::string & v ) {
        Base::stid(trimMulcaId(v));
    }

    void addLid(const Lid& v) {
        const auto & label = getLabel(v);
        if (label.type() == Label::Type::spamDefense) {
            Base::addType(typeFromLabel(label));
        }
        Base::addLabelID(label.lid());
    }

    EnvelopeFactory & fromReflection(reflection::Envelope v) {
        reset();
        for (auto& x : v.attaches) {
            addAttach(std::move(x));
        }
        for (auto& x : v.attributes) {
            addAttribute(MessageAttributes::fromString(x));
        }
        for (const auto& x : v.lids) {
            addLid(std::to_string(x));
        }
        for (const auto& x : v.recipients) {
            addRecipient(x);
        }
        if (v.in_reply_to) {
            inReplyTo(v.in_reply_to.get());
        }
        if (v.thread_count) {
            threadCount(v.thread_count.get());
        }
        if (v.new_count) {
            newCount(PGG_NUMERIC_CAST(std::size_t, v.new_count.get()));
        }
        if (v.attaches.size() > 0) {
            Base::addLabelID(getFakeLabelId(Sym::attached_label));
        }
        date(v.hdr_date);
        extraData(std::move(v.extra_data));
        fid(std::to_string(v.fid));
        firstline(std::move(v.firstline));
        imapId(std::to_string(v.imap_id));
        mid(std::to_string(v.mid));
        receiveDate(v.received_date);
        rfcId(std::move(v.hdr_message_id));
        setDeleted(v.deleted);
        setRecent(v.recent);
        setSeen(v.seen);
        size(PGG_NUMERIC_CAST(std::size_t, v.size));
        stId(std::move(v.st_id));
        subject(std::move(v.subject));
        threadId(v.tid ? std::to_string(*v.tid) : std::string());
        revision(static_cast<Revision::Value>(v.revision));
        if (v.tab) {
            tab(Tab::Type::fromString(v.tab.get(), std::nothrow));
        } else {
            tab(std::nullopt);
        }
        return *this;
    }

private:

    const Label & getLabel( const Lid& lid ) const {
        const auto it = labels.find(lid);
        if(it == labels.end()) {
            throw system_error{error_code{macs::error::noSuchLabel, "unknown lid from database: " + lid}};
        }
        return it->second;
    }

    const LabelSet & labels;
};

inline auto makeEnvelope(const LabelSet& labels, reflection::Envelope v) {
    return EnvelopeFactory(labels).fromReflection(std::move(v)).release();
}

inline auto makeDeletedEnvelope(const LabelSet& labels, reflection::Envelope v) {
    return EnvelopeFactory(labels).fromReflection(std::move(v)).releaseDeleted();
}

inline auto makeEnvelopeWithMime(const LabelSet& labels, reflection::EnvelopeWithMime v) {
    auto e = EnvelopeFactory(labels).fromReflection(std::move(v.envelope)).release();
    auto m = v.mime ? toMime(std::move(*v.mime)) : MimeParts();
    return std::make_tuple(std::move(e), std::move(m));
}

} // namespace pg
} // namespace macs
