#ifndef INCLUDE_INTERNAL_ENVELOPE_QUICK_SAVE_MESSAGE_H_
#define INCLUDE_INTERNAL_ENVELOPE_QUICK_SAVE_MESSAGE_H_

#include <internal/reflection/quick_save_message.h>
#include <internal/envelope/envelope.h>

namespace macs {
namespace pg {

using PGEnvelope = macs::pg::query::Envelope;

struct QuickSaveApplicable {
    struct Result {
        bool applicable;
        std::string message;
        Result(bool applicable = false, std::string msg = "")
        : applicable(applicable), message(std::move(msg)) {}
        operator bool () const { return applicable; }
    };

    static Result applicable() { return Result(true); }
    static Result notApplicable( std::string msg ) {
        return Result(false, std::move(msg));
    }

    Result operator ()(const PGEnvelope & old, const PGEnvelope & newer) const {
        const auto & oldTraits(old.flagsAttributesAndLids());
        const auto & newTraits(newer.flagsAttributesAndLids());

        auto oldLids = oldTraits.lids().value();
        auto newLids = newTraits.lids().value();
        const bool lidsChanged = !boost::equal(boost::sort(newLids), boost::sort(oldLids));
        if(lidsChanged) {
            return notApplicable("labels are changed {"
                    + boost::join(oldLids, ",") + "} => {"
                    + boost::join(newLids, ",") + "}");
        }

        const auto & oldFlags = oldTraits.flags();
        const auto & newFlags = newTraits.flags();
        if(oldFlags.seen() != newFlags.seen()) {
            return notApplicable("\"seen\" changed "
                    + std::to_string(oldFlags.seen()) + "=>"
                    + std::to_string(newFlags.seen()));
        }
        if(oldFlags.deleted() != newFlags.deleted()) {
            return notApplicable("\"deleted\" changed "
                    + std::to_string(oldFlags.deleted()) + "=>"
                    + std::to_string(newFlags.deleted()));
        }
        if(newer.macs().subject() != old.macs().subject()) {
            return notApplicable("subject changed "
                    + old.macs().subject() + "=>" + newer.macs().subject());
        }
        if(newer.macs().threadId() != old.macs().threadId()) {
            return notApplicable("threadId changed "
                    + old.macs().threadId() + "=>" + newer.macs().threadId());
        }
        if(newer.macs().fid() != old.macs().fid()) {
            return notApplicable("fid changed "
                    + old.macs().fid() + "=>" + newer.macs().fid());
        }
        if(newer.macs().mid() != old.macs().mid()) {
            return notApplicable("mid changed "
                    + old.macs().mid() + "=>" + newer.macs().mid());
        }

        return applicable();
    }
};

inline QuickSaveApplicable::Result quickSaveApplicable(const PGEnvelope & old,
        const PGEnvelope & newer) {
    return QuickSaveApplicable()(old, newer);
}

} // namespace pg
} // namespace macs

#endif /* INCLUDE_INTERNAL_ENVELOPE_QUICK_SAVE_MESSAGE_H_ */
