#ifndef INCLUDE_INTERNAL_RECIPIENT_RECIPIENT_H_114222042015
#define INCLUDE_INTERNAL_RECIPIENT_RECIPIENT_H_114222042015

#include <pgg/enumeration.h>
#include <internal/reflection/recipient.h>
#include <string>
#include <vector>

namespace macs {
namespace pg {

class Recipient {
public:
    struct _Type {
        enum Enum {
            unknown,
            from,
            to,
            replyTo,
            cc,
            bcc,
            replyToAll,
            sender
        };
        template <typename Map>
        void fill(Map & m) const {
            auto insert = [&](Enum e, const std::string & v) {
                m.insert(typename Map::value_type(e, v));
            };
            insert(from, "from");
            insert(to, "to");
            insert(replyTo, "reply-to");
            insert(cc, "cc");
            insert(bcc, "bcc");
            insert(replyToAll, "reply-to-all");
            insert(sender, "sender");
            insert(unknown, "unknown");
        }
        typedef _Type Filler;
    };
    using Type = pgg::Enumeration<_Type>;
    using Address = std::string;
    using Name = std::string;
    using Tuple = boost::tuple<Type, Name, Address>;
    using Reflection = reflection::Recipient;

    const Name & name() const { return boost::get<1>(tuple()); }
    const Address & address() const { return boost::get<2>(tuple()); }
    const Type & type() const { return boost::get<0>(tuple()); }
    const Tuple & tuple() const { return t; }

    Recipient() = default;
    Recipient(const Name & name, const Address & address, const Type & type)
    : t(type, name, address) {
    }
    Recipient(Tuple t) : t(std::move(t)) {}
    Recipient(Reflection r) : t(Type::fromString(r.type), std::move(r.name), std::move(r.email)) {}
private:
    Tuple t;
};

inline std::string emailString( const Recipient & r ) {
    if (r.name().empty()) {
        return r.address();
    } else if (r.address().empty()) {
        return r.name();
    } else {
        return r.name() + " <" + r.address() + ">";
    }
}

std::vector<Recipient> & fromStringWithType( const std::string & src,
        Recipient::Type type, std::vector<Recipient> & out );

} // namespace pg
} // namespace macs

#endif /* INCLUDE_INTERNAL_RECIPIENT_RECIPIENT_H_114222042015 */
