#pragma once

#include <macs/labels_repository.h>
#include <macs/folders_repository.h>
#include <macs/envelopes_repository.h>
#include <macs/tabs_repository.h>
#include <internal/not_implemented.h>
#include <pgg/query/repository.h>
#include <internal/query/ids.h>
#include <pgg/query/ranges.h>
#include <internal/query/comment.h>
#include <pgg/share.h>
#include <internal/envelope/envelope.h>

namespace macs {
namespace pg {

typedef query::FolderId FID;
typedef query::LabelId LID;
typedef query::UserId UID;
typedef query::MailId MID;
typedef query::StorageId STID;
typedef query::ThreadId TID;
typedef query::RowFrom Row;
typedef query::RowCount Count;
typedef pgg::query::DateRange DateRange;
typedef query::MailReference MailRef;
typedef query::PartId HID;

typedef query::MailIdList MIDS;
typedef query::FolderIdList FIDS;
typedef query::LabelIdVector LIDS_VECTOR;
typedef query::ThreadIdVector TIDS_VECTOR;
typedef query::LabelIdList LIDS;
typedef query::OldLabelIdList OLD_LIDS;

typedef query::Subject Subject;
typedef query::From From;

/// Envelopes access and manipulation class
template <typename DatabaseGenerator>
class EnvelopesRepository : public macs::EnvelopesRepository {
public:
    typedef macs::EnvelopesRepository Base;
    EnvelopesRepository(UserJournalPtr journal, const DatabaseGenerator& generator,
            LabelsRepositoryPtr labels, FoldersRepositoryPtr folders, TabsRepositoryPtr tabs,
            const std::string & uid, pgg::query::RepositoryPtr queryRepository,
            const pgg::RequestInfo& requestInfo, pgg::Milliseconds transactionTimeout)
    : Base(journal), labels_(labels), folders_(folders), tabs_(tabs), uid_(uid),
      queryRepository_(queryRepository), db(generator),
      requestInfo_(requestInfo), transactionTimeout_(transactionTimeout)
    {}

    virtual ~EnvelopesRepository() { }

protected:
    /// get messages by thread
    virtual uint64_t syncListInThread(
            const EnvelopesSorting & order,
            const std::string & thread,
            const boost::variant<std::size_t, string> & message,
            const std::size_t count,
            OnEnvelopeReceive handler) const override;

    virtual void syncListInThreadWithLabel(
            const std::string& threadId,
            const Label& label,
            std::size_t from,
            std::size_t count,
            OnEnvelopeReceive handler
            ) const override;

    virtual void syncListInThreadWithoutLabel(
            const std::string& threadId,
            const Label& label,
            std::size_t from,
            std::size_t count,
            OnEnvelopeReceive handler
            ) const override;

    /// get messages by folder
    void syncListInFolders(
            const EnvelopesSorting & order,
            const boost::variant<std::size_t, string> & message,
            const std::size_t count,
            const std::list<string>& folders,
            const std::list<string>& labels,
            const std::list<string>& excludeLabels,
            bool groups_only,
            std::time_t dateFrom,
            std::time_t dateTo,
            OnEnvelopeReceive handler ) const;

    /// get messages by whole mailbox
    void syncListInMailbox(
            const EnvelopesSorting & order,
            const boost::variant<std::size_t, string> & message,
            const std::size_t count,
            const std::list<string> & labels,
            const std::list<string> & excludeLabels,
            bool groups_only,
            OnEnvelopeReceive handler ) const;

    /// get messages for search
    virtual uint64_t syncListFilterSearch(
            bool unread,
            bool atta,
            const std::list<std::string>& fids,
            const std::list<std::string>& lids,
            const std::list<std::string>& mids,
            const std::string& folderSet,
            const EnvelopesSorting& order,
            OnEnvelopeReceive handler) const override;

    /// get messages with no answer
    virtual uint64_t syncListWithNoAnswer(const std::string& msgId,
                                          const std::string& includeLid,
                                          OnEnvelopeReceive handler) const override;

    /// get replies messages for given
    virtual uint64_t syncListInReplyTo(const std::string& msgId,
                                       OnEnvelopeReceive handler) const override;

    virtual uint64_t syncListInReplyToByMid(const std::string& mid,
                                            OnEnvelopeReceive handler) const override;

    virtual void syncGetFirstEnvelopeDate(const Fid& fid, OnUnixTime handler) const override;
    virtual void syncGetFirstEnvelopeDateOptional(const Fid& fid, OnUnixTimeOptional handler) const override;
    virtual void syncGetFirstEnvelopeDateInTab(const Tab::Type& tab, OnUnixTimeOptional handler) const override;

    virtual void syncUpdateMailAttributes(const std::string& mid,
                                          const std::vector<std::string> attributes,
                                          OnExecute hook) const override;

    virtual void syncGetEnvelopesCount(const std::string& folderId,
                                       std::time_t dateFrom,
                                       std::time_t dateTo,
                                       OnCountReceive) const override;

    /// erase message
    virtual void syncErase(
            const std::list<std::string>& mids,
            macs::OnUpdate hook) const override;

    /// force erase message
    virtual void syncForceErase(
            const std::list<std::string>& mids,
            macs::OnUpdate hook) const override;

    /// add labels to messages
    virtual void syncAddLabels(
            const std::list<Label>& labels,
            const std::list<std::string>& mids,
            OnUpdate hook) const override;

    /// add labels to messages in threads
    virtual void syncAddLabelsByThreads(
            const std::list<Label>& labels,
            const std::list<std::string>& tids,
            OnUpdateMessages hook) const override;

    /// remove label from messages
    virtual void syncRemoveLabels(
            const std::list<Label>& labels,
            const std::list<std::string>& mids,
            OnUpdate hook) const override;

    /// remove label from messages in threads
    virtual void syncRemoveLabelsByThreads(
            const std::list<Label>& labels,
            const std::list<std::string>& tids,
            OnUpdateMessages hook) const override;

    /// change labels of messages
    virtual void syncChangeLabels(const std::list<Mid>& mids,
                                  const std::list<Label>& labelsToAdd,
                                  const std::list<Label>& labelsToDel,
                                  OnUpdate hook) const override;

    /// get messages by set of ids
    void syncEntriesByIds(const std::list<Mid>& mids,
            Hook<Sequence<Envelope>> handler) const override final;

    /// insert new message entry
    void storeMessage(Envelope envelope, macs::MimeParts mime, ThreadMetaProvider p,
            SaveOptions saveOptions, OnUpdateEnvelope hook) const;

    void syncInsertEntry(Envelope envelope, macs::MimeParts mime, ThreadMetaProvider p,
            SaveOptions saveOptions, OnUpdateEnvelope hook) const override;
    void syncUpdateEntry(const Envelope& old, Envelope newer, macs::MimeParts mime,
            ThreadMetaProvider p, bool ignoreDuplicates, OnUpdateEnvelope hook) const override;

    virtual void syncResetFreshCounter(macs::OnUpdate hook) const override;

    virtual void syncMoveMessages(
            const std::string& destFolder,
            const std::optional<Tab::Type>& destTab,
            const std::list<std::string>& mids,
            macs::OnUpdate hook) const override;

    virtual void syncCopyMessages(const std::string& dstFid,
            const std::vector<std::string>& mids, OnCopy hook) const override;

    virtual void syncUpdateStatus(const std::list<std::string>& mids,
            Envelope::Status status,
            macs::OnUpdate hook) const override;

    virtual void syncGetMidsByFolder(const Fid& fid, OnMidsReceive handler) const override;
    virtual void syncGetMidsRangeByFolder(const Fid& fid, const std::optional<Mid>& fromMid, size_t count,
            OnMidsReceive handler) const override;
    virtual void syncGetMidsByFolderWithoutStatus(const Fid& fid,
        Envelope::Status excludeStatus, OnMidsReceive handler) const override;
    virtual void syncGetMidsRangeByFolderWithoutStatus(const Fid& fid, Envelope::Status excludeStatus,
            const std::optional<Mid>& fromMid, std::size_t count, OnMidsReceive handler) const override;

    virtual void syncGetMidsByTab(const Tab::Type& tab, OnMidsReceive handler) const override;
    virtual void syncGetMidsByTabWithoutStatus(const Tab::Type& tab,
        Envelope::Status excludeStatus, OnMidsReceive handler) const override;

    virtual void syncGetMidsByConditions(const EnvelopesQueryMidsByConditions& conds,
                                         OnMidsReceive hook) const override;
    virtual void syncGetMidsByConditionsWithoutStatus(const EnvelopesQueryMidsByConditions& conds,
                                                      Envelope::Status excludeStatus,
                                                      OnMidsReceive hook) const override;

    virtual void syncGetMidsWithoutStatus(const MidList& mids,
                                          Envelope::Status excludeStatus,
                                          OnMidsReceive handler) const override;

    virtual void syncGetMidsByTidsAndLids(const Tids& tids, const Lids& lids,
            OnMidsReceive handler) const override;

    virtual void syncCheckDuplicates(const Envelope &,
            CheckDuplicates) const override;

    virtual void syncGetAttachments(const Mid& mid,
                                    OnAttachmentsReceive handler) const override;

    virtual void syncGetMessageStIds(const Mids& mids,
                                     OnMidsWithStidsReceive hook) const override;

    virtual void syncGetWindatMessageStId(const Mid& mid, const Hid& hid,
                                          OnStIdReceive hook) const override;

    virtual void syncGetMimes(const Mids& mids, OnMidsWithMimes hook) const override;
    virtual void syncGetMimesWithDeleted(const Mids& mids, OnMidsWithMimes hook) const override;
    virtual void syncGetWindatMimes(const Mids& mids, OnMidsWithMimes hook) const override;

    virtual void syncGetMimesWithAttaches(const Mids& mids, OnMidsWithMimesAndAttaches hook) const override;
    virtual void syncGetMimesWithAttachesWithDeleted(const Mids& mids, OnMidsWithMimesAndAttaches hook) const override;

    virtual void syncDeleteFromStorage(const Stid& stid, macs::OnExecute hook) const override;

    virtual uint64_t syncThreadsByTids(const TidVector& tids,
                                       OnEnvelopeReceive handler) const override;

    virtual uint64_t syncThreadsWithLabel(const std::string& label,
                                          size_t from,
                                          size_t count,
                                          OnEnvelopeReceive handler) const override;
    virtual void syncThreadsInFolderWithoutLabels(const std::string& folder,
                                                  const std::list<std::string>& labels,
                                                  size_t from,
                                                  size_t count,
                                                  OnEnvelopeReceive handler) const override;
    virtual void syncThreadsInFolderWithLabels(const std::string& folder,
                                               const std::list<std::string>& labels,
                                               size_t from,
                                               size_t count,
                                               OnEnvelopeReceive handler) const override;
    virtual void syncMessagesInFolderWithoutLabels(const std::string& folder,
                                                   const std::list<std::string>& labels,
                                                   size_t from,
                                                   size_t count,
                                                   const EnvelopesSorting& order,
                                                   OnEnvelopeReceive handler) const override;
    virtual void syncMessagesInFolderWithLabels(const std::string& folder,
                                                const std::list<std::string>& labels,
                                                size_t from,
                                                size_t count,
                                                const EnvelopesSorting& order,
                                                OnEnvelopeReceive handler) const override;

    virtual void syncGetMessageId(const Mid& mid, OnRfcMessageId handler) const override;
    virtual void syncGetFreshCounter(OnCountReceive handler) const override;
    virtual void syncGetAttachesCounters(OnAttachesCounters handler) const override;

    virtual void syncEnvelopesQueryInMailbox(const std::string& fromMid,
                                             size_t from, size_t count, bool groups,
                                             const EnvelopesSorting& order,
                                             const std::list<std::string>& labels,
                                             const std::list<std::string>& excludeLabels,
                                             OnEnvelopeReceive handler) const override;

    virtual void syncEnvelopesQueryInFolder(const EnvelopesQueryInFolder::Params& params,
                                            OnEnvelopeReceive handler) const override;

    virtual void syncGetByMessageId(const Fid& fid,
                                    const std::string& messageId,
                                    OnMidsAndImapIds handler) const override;

    virtual void syncGetByMessageId(const RfcMessageId& messageId,
                                    const FidVec& excludedFids,
                                    OnMidsReceive handler) const override;

    virtual void syncEnvelopesInFolderWithMimes(size_t from, size_t count,
                                             const EnvelopesSorting& order, const Fid& fid,
                                             OnEnvelopeWithMimeReceive handler) const override;

    virtual void syncEnvelopesByMidsWithMimes(const MidList& mids,
                                             OnEnvelopeWithMimeReceive handler) const override;

    void syncMidsByThreadAndWithSameHeaders(const std::string& threadId, OnMidsReceive handler) const override;

    void syncMidsByHdrDateAndMessageIdPairs(const EnvelopesQueryMidsByHdrPairs& hdrPairs,
                                            OnMidsReceive handler) const override;

    virtual void syncEnvelopesInFolderWithMimesByChunks(
            size_t minImapId, size_t maxImapId, size_t chunkSize,
            const Fid &fid, OnEnvelopeWithMimeChunkReceive handler) const override;

    virtual void syncGetDeletedMessages(size_t from, size_t count,
                                        OnEnvelopeReceive handler) const override;

    virtual void syncGetDeletedMessagesInInterval(size_t from, size_t count,
                                                  std::time_t dateFrom, std::time_t dateTo,
                                                  OnEnvelopeReceive handler) const override;

    virtual void syncGetNewCountInFolderWithLabels(const Fid& fid,
                                                   const std::list<Lid>& labels,
                                                   size_t limit,
                                                   OnCountReceive handler) const override;

    virtual void syncGetNewCountInFolderWithoutLabels(const Fid& fid,
                                                      const std::list<Lid>& labels,
                                                      size_t limit,
                                                      OnCountReceive handler) const override;

    virtual void syncEnvelopesQueryInTab(const EnvelopesQueryInTab::Params& params,
                                         OnEnvelopeReceive handler) const override;

    virtual void syncCountNewMessagesWithLids(const std::list<Lid>& labels,
                                              size_t limit,
                                              OnCountReceive handler) const override;

    virtual void syncGetMidsByLabel(const Label& label, OnMidsReceive hook) const override;

protected:
    template<typename Query>
    void syncChangeLabels(
        const std::list<Label>& labels,
        const std::list<std::string>& mids,
        OnUpdate hook
    ) const;

    template<typename Query>
    void syncChangeLabelsByThreads(
        const std::list<Label>& labels,
        const std::list<std::string>& tids,
        OnUpdateMessages hook
    ) const;

    template <typename Query>
    void syncGetNewCountByFolderAndLabels(
        const Fid& fid,
        const std::list<Lid>& labels,
        size_t limit,
        OnCountReceive handler
    ) const;

private:
    template <typename QueryT, typename HandlerT>
    void request(const QueryT & query, HandlerT handler) const;
    template <typename QueryT, typename HandlerT>
    void requestWithMimes(const QueryT & query, HandlerT handler) const;

    std::shared_ptr<const EnvelopesRepository> getSelf() const {
        return std::dynamic_pointer_cast<const EnvelopesRepository>(pgg::share(this));
    }

    const std::string & uid() const {
        return uid_;
    }
    macs::LabelsRepository & labels() const {
        return *labels_;
    }

    macs::FoldersRepository & folders() const {
        return *folders_;
    }

    const macs::TabsRepository& tabs() const {
        return *tabs_;
    }

    template <typename Handler>
    void getAllLabels(Handler h) const {
        labels().getAllLabels(std::move(h));
    }

    template <typename Handler>
    void getAllFolders(Handler h) const {
        folders().getAllFolders(std::move(h));
    }

    template <typename Handler>
    void getAllTabs(Handler h) const {
        tabs().getAllTabs(std::move(h));
    }

    const pgg::query::Repository & queryRepository() const {
        return *queryRepository_;
    }
    template <typename T, typename ...ArgsT >
    T query(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                std::forward<ArgsT>(args) ...);
    }

    template <typename T, typename ...ArgsT >
    T queryUpdate(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                requestInfo_, std::forward<ArgsT>(args) ...);
    }

    LabelsRepositoryPtr labels_;
    FoldersRepositoryPtr folders_;
    TabsRepositoryPtr tabs_;
    const std::string uid_;
    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
    const pgg::RequestInfo requestInfo_;
    pgg::Milliseconds transactionTimeout_;
};

template <typename DatabaseGenerator>
EnvelopesRepositoryPtr createEnvelopesRepository(
    macs::UserJournalPtr journal,
    DatabaseGenerator dbg,
    LabelsRepositoryPtr labelsRepository,
    FoldersRepositoryPtr foldersRepository,
    TabsRepositoryPtr tabsRepository,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    const pgg::RequestInfo& requestInfo,
    pgg::Milliseconds timeout);

template <typename Labels>
auto createChangeLabelsParams(Labels&& labels, bool boolFlag) {
    query::LabelIdList lids;

#ifndef __clang__
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wmaybe-uninitialized"
#endif

    query::SetSeen seen;
    query::SetRecent recent;
    query::SetDeleted deleted;

#ifndef __clang__
#pragma GCC diagnostic pop
#endif

    for(const auto& l : labels) {
        if (l.symbolicName() == Label::Symbol::seen_label) {
            seen.value = boolFlag;
        } else if (l.symbolicName() == Label::Symbol::recent_label) {
            recent.value = boolFlag;
        } else if (l.symbolicName() == Label::Symbol::deleted_label) {
            deleted.value = boolFlag;
        } else {
            lids.value.push_back(l.lid());
        }
    }

    return std::make_tuple(std::move(lids), seen, recent, deleted);
}


} // namespace pg
} // namespace macs

#include "repository.ipp"
