#include <future>
#include <pgg/chrono.h>
#include <internal/envelope/repository.h>
#include <internal/envelope/query.h>
#include <internal/envelope/convert.h>
#include <internal/envelope/status_to_label.h>
#include <internal/envelope/spam_defence_type.h>
#include <internal/envelope/quick_save_message.h>
#include <boost/variant/variant.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <boost/range.hpp>
#include <pgg/error.h>
#include <pgg/range.h>
#include <pgg/numeric_cast.h>
#include <boost/date_time/posix_time/conversion.hpp>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <macs/mail_reference.h>
#include <internal/hooks/wrap.h>
#include <internal/reflection/attaches_counters.h>
#include <internal/reflection/copy_messages.h>
#include <internal/reflection/count.h>
#include <internal/reflection/first_date.h>
#include <internal/reflection/message_by_st_id.h>
#include <internal/reflection/mid.h>
#include <internal/reflection/stid_by_mid.h>
#include <internal/reflection/store_message.h>
#include <internal/reflection/mid_and_imap_id.h>
#include <internal/reflection/stid.h>
#include <internal/reflection/mime_by_mid.h>
#include <internal/reflection/mime_and_attach_by_mid.h>
#include <internal/envelope/store_message_transaction.h>
#include <internal/hooks/on_check_duplicates.h>
#include <internal/label/fake.h>
#include <internal/envelope/move_message.h>

namespace macs {
namespace pg {

namespace detail {

struct is_str : public boost::static_visitor<int> {
    bool operator() ( int ) const { return false; }
    bool operator() ( const std::string & ) const { return true; }
};

inline bool needNearest(const boost::variant<std::size_t, string> & message) {
    return boost::apply_visitor ( is_str(), message );
}

template <class LidRange>
inline bool needWithLabel(const LidRange & labels,
        const LidRange & excludeLabels) {
    return excludeLabels.empty() && !labels.empty() && !isFake(labels.front());
}

template <class LidRange>
inline bool needOnlyNew(const LidRange & excludeLabels) {
    const auto lid = getFakeLabelId(Label::Symbol::seen_label);
    return std::find(excludeLabels.begin(), excludeLabels.end(), lid) != excludeLabels.end();
}

template <class LidRange>
inline bool needWithAttaches(const LidRange & labels) {
    const auto lid = getFakeLabelId(Label::Symbol::attached_label);
    return std::find(labels.begin(), labels.end(), lid) != labels.end();
}

template <class LidRange>
inline bool needSpam(const LidRange & labels) {
    const auto lid = getFakeLabelId(Label::Symbol::spam_label);
    return std::find(labels.begin(), labels.end(), lid) != labels.end();
}

template <class LidRange>
inline bool needSynced(const LidRange & labels) {
    const auto lid = getFakeLabelId(Label::Symbol::synced_label);
    return std::find(labels.begin(), labels.end(), lid) != labels.end();
}

inline bool needOnlyOneFolder(const std::list<string>& folders) {
    return folders.size() == 1;
}

inline auto createFakeLabelsParams(const Label::Symbol label) {
    const query::IsAttachedFilter attached = (label == Label::Symbol::attached_label);
    const query::IsSpamFilter spam = (label == Label::Symbol::spam_label);
    const query::IsSyncedFilter synced = (label == Label::Symbol::synced_label);

    return std::make_tuple(attached, spam, synced);
}

} // namespace detail

template <typename DatabaseGenerator>
template <typename QueryT, typename HandlerT>
void EnvelopesRepository<DatabaseGenerator>::request(const QueryT & query, HandlerT handler) const {
    getAllLabels([db = db(), h = std::move(handler), query] (error_code e, LabelSet labels) mutable {
        if(e) {
            h(e);
        } else {
            db->request(query, wrapHook<reflection::Envelope>(std::move(h),
                    [labels=std::move(labels)](auto v){
                        return makeEnvelope(labels, std::move(v));
            }));
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncListInMailbox(
        const EnvelopesSorting & order,
        const boost::variant<std::size_t, string> & message,
        const std::size_t countArg,
        const std::list<string> & labels,
        const std::list<string> & excludeLabels,
        bool /* groups_only */,
        OnEnvelopeReceive handler) const {

    const Count count = countArg;
    if ( detail::needNearest(message) ) {
        const MID mid = boost::get< std::string >(message);
        request(query<query::MailboxListNearest>(count, mid), std::move(handler));
    } else {
        const Row row = boost::get<std::size_t>(message);
        const auto args = std::make_tuple(count, row, order);
        if ( detail::needWithLabel(labels, excludeLabels) ) {
            const LID lid( labels.front() );
            request(query<query::MailboxListLabelReceivedDateSort>(args, lid), std::move(handler));
        } else if ( detail::needOnlyNew(excludeLabels) ) {
            request(query<query::MailboxListOnlyNew>(args), std::move(handler));
        } else if ( detail::needWithAttaches(labels) ) {
            auto fakeLabelsParams = detail::createFakeLabelsParams(Label::Symbol::attached_label);
            request(query<query::MailboxListAllWithFakeLabel>(args, fakeLabelsParams), std::move(handler));
        } else if ( detail::needSpam(labels) ) {
            auto fakeLabelsParams = detail::createFakeLabelsParams(Label::Symbol::spam_label);
            request(query<query::MailboxListAllWithFakeLabel>(args, fakeLabelsParams), std::move(handler));
        } else if ( detail::needSynced(labels) ) {
            auto fakeLabelsParams = detail::createFakeLabelsParams(Label::Symbol::synced_label);
            request(query<query::MailboxListAllWithFakeLabel>(args, fakeLabelsParams), std::move(handler));
        } else {
            request(query<query::MailboxListAll>(args), std::move(handler));
        }
    }
}

template <typename DatabaseGenerator>
uint64_t EnvelopesRepository<DatabaseGenerator>::syncListInThread(const EnvelopesSorting & order,
        const std::string & thread,
        boost::variant<std::size_t, string> const & message,
        const std::size_t countArg,
        OnEnvelopeReceive handler) const {

    const Count count = countArg;
    if ( !thread.empty() ) {
        if ( detail::needNearest(message) ) {
            const MID mid = boost::get< std::string >(message);
            request(query<query::MailboxListNearestInThread>(count, mid, order), std::move(handler));
        } else {
            const Row rowFrom = boost::get<std::size_t>(message);
            request(query<query::MailboxListThread>(rowFrom, count, TID(thread), order), std::move(handler));
        }
    }

    return 0;
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncListInThreadWithLabel(const std::string& threadId,
                                                    const Label& label,
                                                    std::size_t from,
                                                    std::size_t count,
                                                    OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(TID(threadId), LID(label.lid()), Row(from), Count(count));
    request(query<query::MailboxListThreadWithLabel>(args), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncListInThreadWithoutLabel(
        const std::string& threadId,
        const Label& label,
        std::size_t from,
        std::size_t count,
        OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(TID(threadId), LID(label.lid()), Row(from), Count(count));
    request(query<query::MailboxListThreadWithoutLabel>(args), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetFirstEnvelopeDate(const Fid& fid, OnUnixTime handler) const {
    using R = reflection::FirstDate;
    auto singleRowHandler = wrapHook<R>(std::move(handler), [](R v){ return v.received_date; });
    auto q = query<query::MailboxGetFirstEnvelopeDate>(FID(fid));
    db()->fetch(q, singleRowHandler);
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetFirstEnvelopeDateOptional(const Fid& fid, OnUnixTimeOptional handler) const {
    using R = reflection::FirstDate;
    auto singleRowHandler = wrapHook<R>(std::move(handler), [](R v){ return v.received_date; });
    auto q = query<query::MailboxGetFirstEnvelopeDate>(FID(fid));
    db()->fetch(q, singleRowHandler);
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetFirstEnvelopeDateInTab(const Tab::Type& tab, OnUnixTimeOptional handler) const {
    using R = reflection::FirstDate;
    auto singleRowHandler = wrapHook<R>(std::move(handler), [](R v){ return v.received_date; });
    auto q = query<query::MailboxGetFirstEnvelopeDateInTab>(query::TabType(tab));
    db()->fetch(q, singleRowHandler);
}

namespace detail {

inline DateRange createLegacyDateRange(std::time_t from, std::time_t to) {
    const auto fromValue = DateRange::getValue(from);
    const auto toValue = to
        ? DateRange::getValue(to)
        : pgg::query::DateRangeTag::max();
    return DateRange(fromValue, toValue);
}

} // namespace detail

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncUpdateMailAttributes(const std::string& mid,
                                                                      const std::vector<std::string> attributes,
                                                                      OnExecute hook) const {
    const auto q = query<query::UpdateMessageAttributes>(query::MailId(mid), attributes);
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetEnvelopesCount(const std::string &folderId,
                                                std::time_t dateFrom,
                                                std::time_t dateTo,
                                                OnCountReceive hook) const {
    auto handler = wrapHook(
        std::move(hook),
        [=](std::ostream & s) {
            s << "syncGetEnvelopesCount(" << folderId << ", "
              << dateFrom << ", " << dateTo << ")";
        }
    );
    auto q = query<query::MailboxGetEnvelopesCount>(FID(folderId), detail::createLegacyDateRange(dateFrom, dateTo));
    db()->fetch(q, handler);
}

template <typename DatabaseGenerator>
uint64_t EnvelopesRepository<DatabaseGenerator>::syncListWithNoAnswer(const std::string& msgId,
                                                   const std::string& includeLid,
                                                   OnEnvelopeReceive handler) const {
    const auto q = query<query::MailboxListByMsgIdWithLabel>(query::MessageId(msgId), LID(includeLid));
    request(q, std::move(handler));
    return 0;
}

template <typename DatabaseGenerator>
uint64_t EnvelopesRepository<DatabaseGenerator>::syncListInReplyTo(const std::string& msgId,
                                                OnEnvelopeReceive handler) const {
    const auto ref = getMailReference(msgId);
    const auto q = query<query::MailboxListInReplyTo>(MailRef(ref));
    request(q, std::move(handler));
    return 0;
}

template <typename DatabaseGenerator>
uint64_t EnvelopesRepository<DatabaseGenerator>::syncListInReplyToByMid(const std::string& mid,
                                                     OnEnvelopeReceive handler) const {
    const auto h = [thiz = getSelf(), hook = std::move(handler)] (error_code ec, Envelope envelope) {
        if (!ec) {
            thiz->syncListInReplyTo(envelope.rfcId(), hook);
        } else {
            hook(ec);
        }
    };
    getById(mid, h);
    return 0;
}

template <typename DatabaseGenerator>
uint64_t EnvelopesRepository<DatabaseGenerator>::syncListFilterSearch(bool unread,
        bool atta,
        const std::list<std::string>& fids,
        const std::list<std::string>& lids,
        const std::list<std::string>& mids,
        const std::string& /*folderSet*/,
        const EnvelopesSorting& order ,
        OnEnvelopeReceive handler) const {

    const auto args = std::make_tuple(MIDS(mids), FIDS(fids), LIDS(lids),
            query::FilterUnread(unread), query::FilterAttaches(atta), order);

    request(query<query::MailboxListFilterSearch>(args), std::move(handler));

    return 0;
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncListInFolders(
        const EnvelopesSorting & order,
        const boost::variant<std::size_t, string> & message,
        const std::size_t countArg,
        const std::list<string>& folders,
        const std::list<string>& /*labels*/,
        const std::list<string>& excludeLabels,
        bool groups_only,
        std::time_t dateFrom,
        std::time_t dateTo,
        OnEnvelopeReceive handler ) const {
    if ( folders.empty() ) {
        throw std::logic_error("macs::pg::EnvelopesRepository::syncListInFolders(): folders are not specified! ");
    }

    if( detail::needNearest(message) ) {
        throw std::logic_error("macs::pg::EnvelopesRepository::syncListInFolders(): query is not supported! ");
    }

    const auto args = std::make_tuple(
            Row{PGG_NUMERIC_CAST(uint64_t, boost::get<std::size_t>(message))},
            Count{PGG_NUMERIC_CAST(uint64_t, countArg)},
            detail::createLegacyDateRange(dateFrom, dateTo) );

    if ( groups_only ) {
        request(query<query::ThreadsView>(args, FID{folders.front()}), std::move(handler));
    } else {
        if ( detail::needOnlyNew(excludeLabels) ) {
            request(query<query::MailboxListFoldersOnlyNew>(args, FIDS{folders}, order), std::move(handler));
        } else {
            if ( detail::needOnlyOneFolder(folders) ) {
                request(query<query::MailboxListOneFolder>(args, FID{folders.front()}, order), std::move(handler));
            } else {
                request(query<query::MailboxListFolders>(args, FIDS{folders}, order), std::move(handler));
            }
        }
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncEntriesByIds(const std::list<std::string>& mids,
        Hook<Sequence<Envelope>> handler) const {
    getAllLabels([self = getSelf(), h = std::move(handler), mids] (error_code e, LabelSet l) mutable {
        if(e) {
            h(std::move(e));
        } else {
            const auto lbls = std::make_shared<LabelSet>(std::move(l));
            using R = reflection::Envelope;
            self->db()->request(self->template query<query::MailboxEntriesByIds>(MIDS(mids)),
                    wrapHook<R>(std::move(h), [lbls] (R v) {
                            return makeEnvelope(*lbls, std::move(v));
                        })
            );
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncMoveMessages(
        const std::string &destFid,
        const std::optional<Tab::Type> &destTab,
        const std::list<std::string> &mids,
        OnUpdate hook) const {
    getAllFolders([self = getSelf(), destFid, destTab, mids, hook = std::move(hook)]
                  (error_code err, FolderSet f) {
        if (err) {
            hook(std::move(err));
        } else {
            auto inboxFid = f.fid(Folder::Symbol::inbox);
            self->getAllTabs([self, destFid, destTab, mids, inboxFid = std::move(inboxFid), hook = std::move(hook)]
                             (error_code err, TabSet t) {
                if (err) {
                    hook(std::move(err));
                } else {
                    auto moveArgs = MoveMessageArgs{mids, destFid, destTab, std::move(inboxFid), std::move(t)};
                    ::macs::pg::moveMessage(self->db(), self->queryRepository_, self->uid(), self->requestInfo_,
                                            std::move(moveArgs), std::move(hook));
                }
            });
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncErase(const std::list<std::string>& midList,
        OnUpdate hook) const {
    auto eraseQuery = queryUpdate<query::MailboxEraseMessages>(MIDS(midList));
    db()->fetch(eraseQuery, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncForceErase(const std::list<std::string>& midList,
                                                            OnUpdate hook) const {
    auto eraseQuery = queryUpdate<query::MailboxForceEraseMessages>(MIDS(midList));
    db()->fetch(eraseQuery, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncResetFreshCounter(OnUpdate hook) const {
    auto q = queryUpdate<query::ResetFreshCounter>();
    db()->fetch(q, wrapHook(std::move(hook)));
}

namespace detail {

template <typename T>
struct BoolFlag;

template <>
struct BoolFlag<query::AddLabels> : public std::true_type {};

template <>
struct BoolFlag<query::RemoveLabels> : public std::false_type {};

template <>
struct BoolFlag<query::AddLabelsByThreads> : public std::true_type {};

template <>
struct BoolFlag<query::RemoveLabelsByThreads> : public std::false_type {};

} // namespace detail


template <typename DatabaseGenerator>
template<typename QueryType>
void EnvelopesRepository<DatabaseGenerator>::syncChangeLabels(const std::list<Label>& labels,
        const std::list<std::string>& midsList,
        OnUpdate hook
    ) const
{
    if (midsList.empty()) {
        throw std::invalid_argument(std::string(__PRETTY_FUNCTION__)
                + " midsList is empty");
    }
    if (labels.empty()) {
        throw std::invalid_argument(std::string(__PRETTY_FUNCTION__)
                + " labels are empty");
    }

    const auto params = createChangeLabelsParams(labels, detail::BoolFlag<QueryType>());
    const auto labelsQuery = queryUpdate<QueryType>(MIDS(midsList), params);
    db()->fetch(labelsQuery, wrapHook(std::move(hook)));
}


/// add labels to messages
template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncAddLabels(
        const std::list<Label>& labels,
        const std::list<std::string>& mids,
        OnUpdate hook) const
{
    syncChangeLabels<query::AddLabels>(labels, mids, std::move(hook));
}

/// remove label from messages
template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncRemoveLabels(const std::list<Label>& labels,
        const std::list<std::string>& mids,
        OnUpdate hook) const
{
    syncChangeLabels<query::RemoveLabels>(labels, mids, std::move(hook));
}

/// change labels of messages
template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncChangeLabels(const std::list<Mid>& mids,
                                           const std::list<Label>& labelsToAdd,
                                           const std::list<Label>& labelsToDel,
                                           OnUpdate hook) const {
    if (mids.empty()) {
        throw std::invalid_argument(std::string(__PRETTY_FUNCTION__)
                + " midsList is empty");
    }
    if (labelsToAdd.empty() && labelsToDel.empty()) {
        throw std::invalid_argument(std::string(__PRETTY_FUNCTION__)
                + " labels are empty");
    }

    auto getLids = [](const std::list<Label>& labels) {
        auto lids = labels | boost::adaptors::transformed([](auto& l){
            return l.lid();
        });
        return std::list<Lid>{std::begin(lids), std::end(lids)};
    };

    const auto query = queryUpdate<query::ChangeLabels>(MIDS(mids),
            LIDS(getLids(labelsToAdd)), OLD_LIDS(getLids(labelsToDel)));
    db()->fetch(query, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
template <class Query>
void EnvelopesRepository<DatabaseGenerator>::syncChangeLabelsByThreads(const std::list<Label>& labels,
        const std::list<std::string>& tids,
        OnUpdateMessages hook) const {
    if (labels.empty()) {
        throw std::invalid_argument(std::string("labels is empty in ") + __PRETTY_FUNCTION__);
    }
    if (tids.empty()) {
        throw std::invalid_argument(std::string("tids is empty in ") + __PRETTY_FUNCTION__);
    }

    auto params = createChangeLabelsParams(labels, detail::BoolFlag<Query>());
    auto labelsQuery = queryUpdate<Query>(query::ThreadIdList(tids), std::move(params));
    db()->fetch(labelsQuery, wrapHook(std::move(hook)));
}

/// add labels to messages in threads
template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncAddLabelsByThreads(const std::list<Label>& labels,
        const std::list<std::string>& tids,
        OnUpdateMessages hook) const {
    syncChangeLabelsByThreads<query::AddLabelsByThreads>(labels, tids, std::move(hook));
}

/// remove label from messages in threads
template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncRemoveLabelsByThreads(const std::list<Label>& labels,
        const std::list<std::string>& tids,
        OnUpdateMessages hook) const {
    syncChangeLabelsByThreads<query::RemoveLabelsByThreads>(labels, tids, std::move(hook));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncCopyMessages(const std::string& dstFid,
        const std::vector<std::string>& mids, OnCopy hook) const {
    using Mids = query::MailIds;

    const auto q = queryUpdate<query::CopyMessages>(FID(dstFid), Mids(mids));

    const auto handleRow = [hook](const apq::row_iterator::value_type& row) {
        const auto v = pgg::cast<reflection::CopyMessages>(row);
        std::vector<Mid> mids;
        for (auto mid : v.mids) {
            mids.emplace_back(std::to_string(mid));
        }
        hook({static_cast<Revision::Value>(v.revision), std::move(mids)});
    };

    const auto handle = [hook, handleRow] (error_code result, auto data) {
        if (result) {
            hook(result);
        } else if(data.empty()){
            hook(error_code(pgg::error::noDataReceived, "No data received from CopyMessages"));
        } else {
            handleRow(data.front());
        }
    };

    db()->fetch(q, handle);
}

template <typename DatabaseGenerator>
uint64_t EnvelopesRepository<DatabaseGenerator>::syncThreadsByTids(const TidVector& tids,
                                                OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(query::ThreadIdVector(tids));
    request(query<query::ThreadsByTids>(args), std::move(handler));
    return 0;
}

template <typename DatabaseGenerator>
uint64_t EnvelopesRepository<DatabaseGenerator>::syncThreadsWithLabel(const std::string& label,
                                                   size_t from,
                                                   size_t count,
                                                   OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(LID(label), Row(from), Count(count));
    request(query<query::ThreadsWithLabel>(args), std::move(handler));
    return 0;
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncThreadsInFolderWithoutLabels(const std::string& folder,
                                                           const std::list<std::string>& labels,
                                                           size_t from,
                                                           size_t count,
                                                           OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(FID(folder), LIDS(labels), Row(from), Count(count));
    request(query<query::ThreadsInFolderWithoutLabels>(args), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncThreadsInFolderWithLabels(const std::string& folder,
                                                        const std::list<std::string>& labels,
                                                        size_t from,
                                                        size_t count,
                                                        OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(FID(folder), LIDS(labels), Row(from), Count(count));
    request(query<query::ThreadsInFolderWithLabels>(args), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncMessagesInFolderWithoutLabels(const std::string& folder,
                                                            const std::list<std::string>& labels,
                                                            size_t from,
                                                            size_t count,
                                                            const EnvelopesSorting& /*order*/,
                                                            OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(FID(folder), LIDS(labels), Row(from), Count(count));
    request(query<query::MessagesInFolderWithoutLabels>(args), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncMessagesInFolderWithLabels(const std::string& folder,
                                                         const std::list<std::string>& labels,
                                                         size_t from,
                                                         size_t count,
                                                         const EnvelopesSorting& /*order*/,
                                                         OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(FID(folder), LIDS(labels), Row(from), Count(count));
    request(query<query::MessagesInFolderWithLabels>(args), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncUpdateStatus(const std::list<std::string>& mids,
                                            Envelope::Status status,
                                            OnUpdate hook) const {
    getAllLabels([self = getSelf(), mids, status, h = std::move(hook)]
                  (error_code e, LabelSet l) mutable {
        if(e) {
            h(std::move(e), NULL_REVISION);
        } else {
            const std::list<Label> labels{ statusToLabel(l, status) };
            if(needToAddLabel(status)) {
                self->syncAddLabels(std::move(labels), mids, std::move(h));
            } else {
                self->syncRemoveLabels(std::move(labels), mids, std::move(h));
            }
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByFolder(const Fid& fid, OnMidsReceive hook) const {
    db()->fetch(query<query::MidsByFolder>(FID(fid)), wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsRangeByFolder(const Fid& fid, const std::optional<Mid>& fromMid,
        size_t count, OnMidsReceive hook) const {
    db()->fetch(query<query::MidsRangeByFolder>(FID(fid), query::OptionalMailId(fromMid), Count(count)),
            wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByFolderWithoutStatus(const Fid& fid,
                                                           Envelope::Status excludeStatus,
                                                           OnMidsReceive hook) const {
    switch (excludeStatus) {
    case Envelope::Status_read:
    case Envelope::Status_unread: {
        const auto q = query<query::MidsByFolderWithSeen>(FID(fid), seenFromStatus(excludeStatus));
        db()->fetch(q, wrapHook(std::move(hook)));
    } break;
    case Envelope::Status_replied:
    case Envelope::Status_forwarded:
        getAllLabels([self = getSelf(), h = wrapHook(std::move(hook)), fid, excludeStatus]
                      (error_code e, LabelSet l) mutable {
            if(e) {
                h(std::move(e), boost::none);
            } else {
                const Label excludeLabel = statusToLabel(l, excludeStatus);
                self->db()->fetch(self->template query<query::MidsByFolderWithoutLabel>(
                        FID(fid), LID(excludeLabel.lid())), std::move(h));
            }
        });
    break;
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsRangeByFolderWithoutStatus(const Fid& fid,
                                                                                   Envelope::Status excludeStatus,
                                                                                   const std::optional<Mid>& fromMid,
                                                                                   size_t count,
                                                                                   OnMidsReceive hook) const {
    switch (excludeStatus) {
        case Envelope::Status_read:
        case Envelope::Status_unread: {
            const auto q = query<query::MidsRangeByFolderWithSeen>(FID(fid), seenFromStatus(excludeStatus),
                    query::OptionalMailId(fromMid), Count(count));
            db()->fetch(q, wrapHook(std::move(hook)));
        } break;
        case Envelope::Status_replied:
        case Envelope::Status_forwarded:
            getAllLabels([self = getSelf(), h = wrapHook(std::move(hook)), fid, excludeStatus, fromMid, count]
                                 (error_code e, LabelSet l) mutable {
                if(e) {
                    h(std::move(e), boost::none);
                } else {
                    const Label excludeLabel = statusToLabel(l, excludeStatus);
                    self->db()->fetch(self->template query<query::MidsRangeByFolderWithoutLabel>(
                            FID(fid), LID(excludeLabel.lid()), query::OptionalMailId(fromMid), Count(count)), std::move(h));
                }
            });
            break;
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByTab(const Tab::Type& tab, OnMidsReceive hook) const {
    db()->fetch(query<query::MidsByTab>(query::TabType(tab)), wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByTabWithoutStatus(const Tab::Type& tab,
                                                        Envelope::Status excludeStatus,
                                                        OnMidsReceive hook) const {
    switch (excludeStatus) {
    case Envelope::Status_read:
    case Envelope::Status_unread: {
        const auto q = query<query::MidsByTabWithSeen>(query::TabType(tab),
                                                       seenFromStatus(excludeStatus));
        db()->fetch(q, wrapHook(std::move(hook)));
    } break;
    case Envelope::Status_replied:
    case Envelope::Status_forwarded:
        getAllLabels([self = getSelf(), h = wrapHook(std::move(hook)), tab, excludeStatus]
                      (error_code e, LabelSet l) mutable {
            if(e) {
                h(std::move(e), boost::none);
            } else {
                const Label excludeLabel = statusToLabel(l, excludeStatus);
                self->db()->fetch(self->template query<query::MidsByTabWithoutLabel>(
                        query::TabType(tab), LID(excludeLabel.lid())), std::move(h));
            }
        });
    break;
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::storeMessage(Envelope e, macs::MimeParts m, ThreadMetaProvider p,
        SaveOptions saveOptions, OnUpdateEnvelope h) const {
    using Transaction = pgg::query::fallback::Transaction<DatabaseGenerator, pgg::database::fallback::rules::Master>;

    getAllLabels([self = getSelf(), e = std::move(e), m = std::move(m), p = std::move(p),
          saveOptions = std::move(saveOptions), h = std::move(h)]
         (error_code err, LabelSet l) mutable
    {
        if (err) {
            h(std::move(err));
        } else {
            self->getAllFolders([self, e = std::move(e), m = std::move(m), p = std::move(p), saveOptions = std::move(saveOptions),
                l = std::move(l), h = std::move(h)]
                (error_code err, FolderSet f) mutable
            {
                if (err) {
                    h(std::move(err));
                } else {
                    self->getAllTabs([self, e = std::move(e), m = std::move(m),
                                     p = std::move(p), saveOptions = std::move(saveOptions),
                                     l = std::move(l), f = std::move(f), h = std::move(h)]
                                     (error_code err, TabSet t) mutable
                    {
                        if (err) {
                            h(std::move(err));
                        } else {
                            auto meta = p ? p(e) : ThreadMeta();
                            auto transaction = boost::make_shared<Transaction>(self->queryRepository_);
                            auto storeMessageTransaction = boost::make_shared<StoreMessageTransaction<decltype(self->db), decltype(transaction)>>(
                                    self->db, transaction, self->queryRepository_, self->uid(), self->requestInfo_,
                                    std::move(e), std::move(m), std::move(meta), std::move(l), std::move(f), std::move(t),
                                    std::move(saveOptions), std::move(h), self->transactionTimeout_);
                            (*storeMessageTransaction)();
                        }
                    });
                }
            });
        }
    });

}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncInsertEntry(Envelope e, macs::MimeParts m, ThreadMetaProvider p,
        SaveOptions saveOptions, OnUpdateEnvelope h) const {
    getAllLabels([self = getSelf(), p = std::move(p), h = std::move(h), e = std::move(e),
                  m = std::move(m), saveOptions = std::move(saveOptions)]
                 (error_code err, LabelSet l) mutable {
        if(err) {
            h(std::move(err));
        } else {
            macs::EnvelopeFactory f(e);
            self->storeMessage(synchronizeData(f, l).release(), std::move(m),
                    std::move(p), std::move(saveOptions), std::move(h));
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncUpdateEntry(const Envelope& oldEnv, Envelope newEnv, macs::MimeParts m,
        ThreadMetaProvider p, bool ignoreDuplicates, OnUpdateEnvelope hook ) const {

    getAllLabels([self = getSelf(), p = std::move(p), hook = std::move(hook), ignoreDuplicates,
                  newEnv = std::move(newEnv), oldEnv, m = std::move(m)] (error_code err, LabelSet l) mutable {

        if(err) {
            return hook(std::move(err));
        }

        macs::EnvelopeFactory oldF(oldEnv);
        macs::EnvelopeFactory newF(newEnv);
        macs::pg::query::Envelope old(synchronizeData(oldF, l).release());
        macs::pg::query::Envelope newer(synchronizeData(newF, l).mid(oldEnv.mid()).release());

        const auto applicable = quickSaveApplicable(old, newer);
        if( !applicable ) {
            self->storeMessage(std::move(newer.macs()), std::move(m), p,
                               SaveOptions{ignoreDuplicates, NotificationMode::on, StoreType::box}, hook);
        } else {
            self->db()->fetch(self->template queryUpdate<query::QuickSaveMessage>(newer, m),
                    [self, hook = std::move(hook), newer, m, p = std::move(p), ignoreDuplicates]
                     (error_code error, auto data ) mutable {
                if( error ) {
                    hook(std::move(error));
                } else if( data.empty() ) {
                    hook(error_code(pgg::error::noDataReceived,
                                    "No data received as a result of QuickSaveMessage"));
                } else {
                    const auto v = pgg::cast<reflection::QuickSaveMessage>(data.front());
                    if(v.updated) {
                        hook(macs::UpdateEnvelopeResult(macs::EnvelopeFactory(newer.macs())
                                .revision(PGG_NUMERIC_CAST(Revision, v.revision))
                                .release(),
                            EnvelopeKind::original));
                    } else {
                        self->storeMessage(std::move(newer.macs()), std::move(m), p,
                            SaveOptions{ignoreDuplicates, NotificationMode::on, StoreType::box}, std::move(hook));
                    }
                }
            });
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMessageStIds(const Mids& mids,
                                              OnMidsWithStidsReceive hook) const {
    db()->fetch(query<query::StidsByMids>(query::MailIdList(mids)), wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMimes(const Mids& mids, OnMidsWithMimes hook) const {
    auto handler = wrapHook<reflection::MimeByMid>(std::move(hook), [] (auto v) {
                return std::make_tuple(std::to_string(v.mid), std::move(v.st_id),
                        v.mime ? toMime(std::move(*v.mime)) : MimeParts());
            });
    db()->fetch(query<query::MimesByMids>(MIDS(mids)), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMimesWithDeleted(const Mids& mids, OnMidsWithMimes hook) const {
    auto handler = wrapHook<reflection::MimeByMid>(std::move(hook), [] (auto v) {
        return std::make_tuple(std::to_string(v.mid), std::move(v.st_id),
            v.mime ? toMime(std::move(*v.mime)) : MimeParts());
    });
    db()->fetch(query<query::MimesByMidsWithDeleted>(MIDS(mids)), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetWindatMimes(const Mids& mids, OnMidsWithMimes hook) const {
    auto handler = wrapHook<reflection::MimeByMid>(std::move(hook), [] (auto v) {
                return std::make_tuple(std::to_string(v.mid), std::move(v.st_id),
                        v.mime ? toMime(std::move(*v.mime)) : MimeParts());
            });
    db()->fetch(query<query::WindatMimesByMids>(MIDS(mids)), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMimesWithAttaches(const Mids& mids, OnMidsWithMimesAndAttaches hook) const {
    const auto q = query<query::MimesWithAttachesByMids>(MIDS(mids));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMimesWithAttachesWithDeleted(const Mids& mids, OnMidsWithMimesAndAttaches hook) const {
    const auto q = query<query::MimesWithAttachesByMidsWithDeleted>(MIDS(mids));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetWindatMessageStId(const Mid& mid, const Hid& hid,
                                                   OnStIdReceive hook) const {
    using R = reflection::Stid;
    auto handler = wrapHook<R>(std::move(hook), [] (R v) { return *v.st_id; });
    db()->fetch(query<query::WindatMessageStId>(MID(mid), HID(hid)), handler);
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncDeleteFromStorage(const Stid& stid,
                                                macs::OnExecute hook) const {
    const auto q = query<query::AddToStorageDeleteQueue>(STID(stid));
    db()->execute(q, std::move(hook));
}

namespace detail {

inline std::tuple<FID, Subject, DateRange, From> conditionsToArgs(const EnvelopesQueryMidsByConditions& conds) {
    pgg::Time toTime;
    if (conds.age()) {
        toTime = pgg::now() - conds.age().get();
    }
    const DateRange dateRange(0, pgg::Clock::to_time_t(toTime));

    const std::string unexistentStr =
        "^&#*@(#&*)(!)@()$#&^&*67896783267839642806&*^#&*@HJKGFHJGDHFXFHDJKLFHJKL@#$^&*";
    const std::string subject = conds.subject() ? conds.subject().get() : unexistentStr;
    const std::string from = conds.from() ? conds.from().get() : unexistentStr;
    return std::make_tuple(FID(conds.fid()), Subject(subject), dateRange, From(from));
}

} // namespace detail

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByConditions(const EnvelopesQueryMidsByConditions& conds,
                                                  OnMidsReceive hook) const {
    const auto q = query<query::GetMidsByConditions>(detail::conditionsToArgs(conds));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByConditionsWithoutStatus(const EnvelopesQueryMidsByConditions& conds,
                                                               Envelope::Status excludeStatus,
                                                               OnMidsReceive hook) const {
    const auto commonArgs = detail::conditionsToArgs(conds);

    switch (excludeStatus) {
        case Envelope::Status_read:
        case Envelope::Status_unread: {
            const auto q = query<query::GetMidsByConditionsWithSeen>(commonArgs, seenFromStatus(excludeStatus));
            db()->fetch(q, wrapHook(std::move(hook)));
        } break;
        case Envelope::Status_replied:
        case Envelope::Status_forwarded:
            getAllLabels([self = getSelf(), commonArgs = std::move(commonArgs),
                          handler = wrapHook(std::move(hook)), excludeStatus]
                          (error_code err, LabelSet l) mutable {
                if(err) {
                    return handler(std::move(err), boost::none);
                }
                const Label excludeLabel = statusToLabel(l, excludeStatus);
                const auto q = self->template query<query::GetMidsByConditionsWithoutLabel>(commonArgs, LID(excludeLabel.lid()));
                self->db()->fetch(q, std::move(handler));
        });
        break;
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsWithoutStatus(const MidList& mids,
                                                   Envelope::Status excludeStatus,
                                                   OnMidsReceive hook) const {
    switch (excludeStatus) {
        case Envelope::Status_read:
        case Envelope::Status_unread: {
            const auto q = query<query::GetMidsWithSeen>(MIDS(mids), seenFromStatus(excludeStatus));
            db()->fetch(q, wrapHook(std::move(hook)));
        } break;
        case Envelope::Status_replied:
        case Envelope::Status_forwarded:
            getAllLabels([self = getSelf(), mids,
                          handler = wrapHook(std::move(hook)), excludeStatus]
                          (error_code err, LabelSet l) mutable {
                if(err) {
                    return handler(std::move(err), boost::none);
                }
                const Label excludeLabel = statusToLabel(l, excludeStatus);
                const auto q = self->template query<query::GetMidsWithoutLabel>(
                        MIDS(mids), LID(excludeLabel.lid()));
                self->db()->fetch(q, std::move(handler));
            });
        break;
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByTidsAndLids(const Tids& tids, const Lids& lids,
        OnMidsReceive handler) const {
    db()->fetch(query<query::GetMidsByTidsAndLids>(TIDS_VECTOR{tids}, LIDS_VECTOR{lids}), wrapHook(std::move(
            handler)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncCheckDuplicates(const Envelope & e, CheckDuplicates h) const {
    const auto q = query<query::FindDuplicates>(
        query::Subject( e.subject() ), query::MessageId( e.rfcId() ), query::HdrDate( e.date() ) );

    OnCheckDuplicatesHandler handler(e.from(), e.fid(), std::move(h));

    db()->fetch(q, handler);
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetAttachments(const Mid& mid, OnAttachmentsReceive h) const {
    getById(mid, [h = std::move(h)] (error_code ec, Envelope envelope) {
        if (!ec) {
            h(envelope.attachments());
        } else {
            h(ec);
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMessageId(const Mid& mid, OnRfcMessageId handler) const {
    getById(mid, [handler = std::move(handler)] (error_code ec, auto envelope) {
        handler(std::move(ec), envelope.rfcId());
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetFreshCounter(OnCountReceive hook) const {
    auto handler = wrapHook(
        std::move(hook),
        [=](std::ostream & s) { s << "syncGetFreshCounter()"; }
    );

    const auto q = query<query::GetFreshCounter>();
    db()->fetch(q, handler);
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetAttachesCounters(OnAttachesCounters hook) const {
    const auto q = query<query::GetAttachesCounters>();
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncEnvelopesQueryInMailbox(const std::string &fromMid,
                                                      size_t from, size_t count, bool groups,
                                                      const EnvelopesSorting &order,
                                                      const std::list<std::string> &labels,
                                                      const std::list<std::string> &excludeLabels,
                                                      OnEnvelopeReceive handler) const {

    if (labels.size() == 1 && groups && !isFake(labels.front())) {
        syncThreadsWithLabel(labels.front(), from, count, std::move(handler));
    } else if (fromMid.empty()) {
        syncListInMailbox( order, from, count, labels, excludeLabels, groups, std::move(handler));
    } else {
        syncListInMailbox( order, fromMid, count, labels, excludeLabels, groups, std::move(handler));
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncEnvelopesQueryInFolder(const EnvelopesQueryInFolder::Params& params,
                                                     OnEnvelopeReceive handler) const
{
    auto hasFakes = [](const std::list<std::string>& labels) {
        return boost::count_if(labels, [](auto& l){ return isFake(l); }) > 0;
    };
    if (params.folders.size() == 1 && !params.excludeLabels.empty() && !hasFakes(params.excludeLabels)) {
        if (params.groups) {
            syncThreadsInFolderWithoutLabels(params.folders.front(), params.excludeLabels,
                                             params.from, params.count, std::move(handler));
        } else {
            syncMessagesInFolderWithoutLabels(params.folders.front(), params.excludeLabels,
                                              params.from, params.count, params.order, std::move(handler));
        }
    } else if (params.folders.size() == 1 && !params.labels.empty() && !hasFakes(params.labels)) {
        if (params.groups) {
            syncThreadsInFolderWithLabels(params.folders.front(), params.labels,
                                          params.from, params.count, std::move(handler));
        } else {
            syncMessagesInFolderWithLabels(params.folders.front(), params.labels,
                                           params.from, params.count, params.order, std::move(handler));
        }
    } else if (params.fromMid.empty()) {
        syncListInFolders(params.order, params.from, params.count, params.folders,
                          params.labels, params.excludeLabels, params.groups,
                          params.timeInterval.first, params.timeInterval.second, std::move(handler));
    } else {
        syncListInFolders(params.order, params.fromMid, params.count, params.folders,
                          params.labels, params.excludeLabels, params.groups,
                          params.timeInterval.first, params.timeInterval.second, std::move(handler));
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetByMessageId(const Fid& fid, const std::string& messageId,
                                             OnMidsAndImapIds handler) const {

    db()->fetch(query<query::GetByMessageId>(FID(fid), query::MessageId(messageId)),
        wrapHook<reflection::MidAndImapId>(std::move(handler), [](auto v) {
                return MidAndImapId(std::to_string(v.mid),
                                std::to_string(v.imap_id));
            }));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetByMessageId(const RfcMessageId& messageId,
                                                                const FidVec& excludedFids,
                                                                OnMidsReceive handler) const {
    db()->fetch(
        query<query::GetByMessageIdExceptExcludedFolders>(
            query::MessageId(messageId),
            query::FolderIdVector(excludedFids)
        ),
        wrapHook(std::move(handler))
    );
}

template <typename DatabaseGenerator>
template <typename QueryT, typename HandlerT>
void EnvelopesRepository<DatabaseGenerator>::requestWithMimes(const QueryT & query, HandlerT handler) const {
    getAllLabels([db = db(), h = std::move(handler), query] (error_code e, LabelSet labels) mutable {
        if(e) {
            h(e);
        } else {
            db->request(query, wrapHook<reflection::EnvelopeWithMime>(std::move(h),
                    [labels=std::move(labels)](auto v){
                        return makeEnvelopeWithMime(labels, std::move(v));
            }));
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncEnvelopesInFolderWithMimes(size_t from, size_t count,
        const EnvelopesSorting& order, const Fid& fid, OnEnvelopeWithMimeReceive handler) const {
    const auto q = query<query::MessagesInFolderWithMimes>(FID(fid), Row(from), Count(count), order);
    requestWithMimes(q, std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncEnvelopesByMidsWithMimes(const MidList& mids, OnEnvelopeWithMimeReceive handler) const {
    const auto q = query<query::MessagesByMidsWithMimes>(mids);
    requestWithMimes(q, std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncMidsByThreadAndWithSameHeaders(const std::string& threadId,
                                                             OnMidsReceive handler) const {
    const auto q = query<query::MidsByThreadAndWithSameHeaders>(TID(threadId));
    db()->fetch(q, wrapHook(std::move(handler)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncMidsByHdrDateAndMessageIdPairs(const EnvelopesQueryMidsByHdrPairs& hdrPairs,
                                                             OnMidsReceive handler) const {
    const auto q = query<query::MidsByHdrDateAndMessageIdPairs>(hdrPairs.pairs());
    db()->fetch(q, wrapHook(std::move(handler)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncEnvelopesInFolderWithMimesByChunks(
        size_t minImapId, size_t maxImapId, size_t chunkSize,
        const Fid &fid, OnEnvelopeWithMimeChunkReceive handler) const {
    const auto q = query<query::MessagesInFolderWithMimesByChunks>(
                FID(fid), query::RangeMin(minImapId), query::RangeMax(maxImapId), Count(chunkSize));
    getAllLabels([db=db(), h=std::move(handler), q] (error_code e, LabelSet labels) {
        if (e) {
            h(e);
        } else {
            db->fetch(q, [h=std::move(h), labels=std::move(labels)]
                          (error_code e, auto data) {
                if (e) {
                    h(e);
                } else {
                    auto res = data | boost::adaptors::transformed([labels=std::move(labels)](auto row) {
                        return makeEnvelopeWithMime(labels, pgg::cast<reflection::EnvelopeWithMime>(row));
                    });
                    h({res.begin(), res.end()});
                }
            });
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetDeletedMessages(
        size_t from, size_t count,
        OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(Row(from), Count(count));
    request(query<query::DeletedMessages>(args), std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetDeletedMessagesInInterval(
        size_t from, size_t count,
        std::time_t dateFrom, std::time_t dateTo,
        OnEnvelopeReceive handler) const {
    const auto args = std::make_tuple(Row(from), Count(count),
                                      detail::createLegacyDateRange(dateFrom, dateTo));
    request(query<query::DeletedMessagesInInterval>(args), std::move(handler));
}

template <typename DatabaseGenerator>
template <typename Query>
void EnvelopesRepository<DatabaseGenerator>::syncGetNewCountByFolderAndLabels(
        const Fid& fid,
        const std::list<Lid>& labels,
        size_t limit,
        OnCountReceive handler) const {
    const auto q = query<Query>(FID(fid), LIDS(labels), Count(limit));
    db()->fetch(q, [h = std::move(handler)](error_code e, auto data) {
        if (e) {
            h(e);
        } else {
            h(boost::size(data));
        }
    });
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetNewCountInFolderWithLabels(
        const Fid& fid,
        const std::list<Lid>& labels,
        size_t limit,
        OnCountReceive handler) const {
    syncGetNewCountByFolderAndLabels<query::NewMidsInFolderWithLabels>(
                fid, labels, limit, std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetNewCountInFolderWithoutLabels(
        const Fid& fid,
        const std::list<Lid>& labels,
        size_t limit,
        OnCountReceive handler) const {
    syncGetNewCountByFolderAndLabels<query::NewMidsInFolderWithoutLabels>(
                fid, labels, limit, std::move(handler));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncEnvelopesQueryInTab(const EnvelopesQueryInTab::Params& params,
                                                  OnEnvelopeReceive handler) const {
    auto hasFakes = [](const std::vector<std::string>& labels) {
        return boost::count_if(labels, [](auto& l){ return isFake(l); }) > 0;
    };

    const auto commonArgs = std::make_tuple(
            query::TabType{params.tabType},
            Row{PGG_NUMERIC_CAST(uint64_t, params.from)},
            Count{PGG_NUMERIC_CAST(uint64_t, params.count)} );

    if ( !params.excludeLabels.empty() && !hasFakes(params.excludeLabels) ) {
        if ( params.groupByThreads ) {
            request(query<query::ThreadsInTabWithoutLabels>(
                        commonArgs, LIDS_VECTOR(params.excludeLabels)), std::move(handler));
        } else {
            request(query<query::MessagesInTabWithoutLabels>(
                        commonArgs, LIDS_VECTOR(params.excludeLabels)), std::move(handler));
        }
    } else if ( !params.labels.empty() && !hasFakes(params.labels) ) {
        if ( params.groupByThreads ) {
            request(query<query::ThreadsInTabWithLabels>(
                        commonArgs, LIDS_VECTOR(params.labels)), std::move(handler));
        } else {
            request(query<query::MessagesInTabWithLabels>(
                        commonArgs, LIDS_VECTOR(params.labels)), std::move(handler));
        }
    } else {
        auto dateRange = detail::createLegacyDateRange(params.timeInterval.first,
                                               params.timeInterval.second);
        if ( params.groupByThreads ) {
            request(query<query::ThreadsInTab>(commonArgs, dateRange), std::move(handler));
        } else {
            if ( detail::needOnlyNew(params.excludeLabels) ) {
                request(query<query::MessagesInTabOnlyNew>(commonArgs, dateRange), std::move(handler));
            } else {
                request(query<query::MessagesInTab>(commonArgs, dateRange), std::move(handler));
            }
        }
    }
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncCountNewMessagesWithLids(
        const std::list<Lid>& labels,
        size_t limit,
        OnCountReceive hook) const {
    const auto q = query<query::CountNewMessagesWithLids>(LIDS(labels), Count(limit));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void EnvelopesRepository<DatabaseGenerator>::syncGetMidsByLabel(const Label& label, OnMidsReceive hook) const {
    const auto& lid = label.lid();
    if ( !isFake(lid) ) {
        db()->fetch(query<query::MidsByLid>(LID(lid)), wrapHook(std::move(hook)));
    } else if ( lid == getFakeLabelId(Label::Symbol::attached_label) ) {
        db()->fetch(query<query::MidsWithAttaches>(), wrapHook(std::move(hook)));
    } else {
        throw std::invalid_argument(std::string(__PRETTY_FUNCTION__)
                + " cannot work with lid " + lid);
    }
}

} // namespace pg
} // namespace macs
