#pragma once

#include <macs/envelopes_repository.h>
#include <pgg/query/transactional.h>
#include <internal/envelope/query.h>
#include <internal/thread/query.h>
#include <internal/thread/threads_chooser.h>
#include <internal/thread/add_message_to_thread.h>
#include <internal/mailish/message_info.h>
#include <pgg/query/ranges.h>
#include <pgg/cast.h>
#include <pgg/range.h>
#include <internal/reflection/revision.h>
#include <internal/reflection/store_message.h>
#include <internal/reflection/find_threads.h>
#include <internal/hooks/on_check_duplicates.h>
#include <internal/query/comment.h>
#include <internal/envelope/factory.h>

#include <iterator>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace macs{
namespace pg{

using PGEnvelope = macs::pg::query::Envelope;
using PGEnvelopeDeleted = macs::pg::query::EnvelopeDeleted;

template <typename ConnProvider, typename TransactionPtr>
class StoreMessageTransaction: public boost::asio::coroutine,
                               public boost::enable_shared_from_this<StoreMessageTransaction<ConnProvider, TransactionPtr>>{
public:
    class TabResolver {
    public:
        TabResolver(TabSet tabs) : tabs_(std::move(tabs)) {}

        std::optional<Tab::Type> operator()(const std::optional<Tab::Type>& tab) const {
            if (tab) {
                auto found = tabs_.find(tab.value());
                if (found != tabs_.end()) {
                    return { found->first };
                } else if (tabs_.exists(Tab::Type::relevant)) {
                    return { Tab::Type::relevant };
                }
            }
            return std::nullopt;
        }

    private:
        TabSet tabs_;
    };

    explicit StoreMessageTransaction (ConnProvider conn, TransactionPtr t, pgg::query::RepositoryPtr qr, const std::string& uid,
            const pgg::RequestInfo& ri, Envelope entry, MimeParts mime, ThreadMeta tm, LabelSet labels, FolderSet folders, TabSet tabs,
            macs::EnvelopesRepository::SaveOptions saveOptions, OnUpdateEnvelope hook, pgg::Milliseconds timeout )
    : connProvider_(conn), transaction_(t), queryRepository_(qr), uid_(uid), requestInfo_(ri), envFactory_(entry),
      mime_(std::move(mime)), meta_(std::move(tm)), labels_(std::move(labels)), folders_(std::move(folders)),
      ignoreDuplicates_(saveOptions.ignoreDuplicates), notificationMode_(saveOptions.notificationMode),
      storeType_(saveOptions.storeType), hook_(hook), timeout_(timeout), tabResolver_(std::move(tabs))
    {
    }

    void operator()() {
        const auto cb = [this, self=this->shared_from_this()](auto err){
            if (err) {
                hook_(std::move(err));
            } else {
                (*this)();
            }
        };

        reenter (this) {
            yield transaction_->begin(connProvider_, std::move(cb), timeout_);

            yield lock();

            if (storeType_ == macs::EnvelopesRepository::StoreType::deleted) {
                yield storeDeletedMessage();
                yield getDeletedMessage();
            } else {
                if (!ignoreDuplicates_) {
                    yield checkDuplicates();
                    if (!mid_.empty()) {
                        yield getStoredMessage();
                        yield transaction_->rollback(cb);
                        return hook_(error_code(),
                                     UpdateEnvelopeResult(std::move(storedMessage_), EnvelopeKind::duplicate));
                    }
                }

                yield addToThread();
                yield storeMessage();
                yield getStoredMessage();
            }

            yield transaction_->commit(cb);

            hook_(macs::error_code(), UpdateEnvelopeResult(std::move( storedMessage_ ), EnvelopeKind::original));
        }
    }

private:

    void lock() {
        using namespace macs::pg::query;
        const auto q = query<LockUserDelivery>();
        const auto hook = [this, self=this->shared_from_this()](auto err){
            if (err) {
                hook_(std::move(err));
            } else {
                (*this)();
            }
        };
        transaction_->execute(q, hook);
    }

    void checkDuplicates() {
        using namespace macs::pg::query;
        const auto q = query<FindDuplicates>(
                Subject( envFactory_.product().subject() ),
                MessageId( envFactory_.product().rfcId() ),
                HdrDate( envFactory_.product().date() ) );
        const auto hook = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleDuplicates(std::move(err), std::move(data));
        };
        OnCheckDuplicatesHandler h(envFactory_.product().from(), envFactory_.product().fid(), hook);
        transaction_->fetch(q, h);
    }

    void addToThread() {
        const auto h = [this, self=this->shared_from_this()](auto err, const JoinThreadResult& res) {
            if (err) {
                hook_(std::move(err));
            } else {
                applyJoinThreadResult(envFactory_, res);
                (*this)();
            }
        };

        const auto& envelope = envFactory_.product();
        auto addMessageToThread = boost::make_shared<AddMessageToThread<decltype(transaction_)>>(
            transaction_, queryRepository_, uid_, requestInfo_, meta_, envFactory_.product().rfcId(),
            labels_, folders_, envelope.labels(), envelope.fid(), envelope.receiveDate(), h);
        (*addMessageToThread)();
    }

    void storeMessage() {
        using namespace macs::pg::query;

        auto destTab = tabResolver_(envFactory_.product().tab());
        envFactory_.tab(std::move(destTab));
        const auto q = queryUpdate<StoreMessage>(
                PGEnvelope( envFactory_.release() ), mime_, meta_,
                QuietFlag(notificationMode_ == macs::EnvelopesRepository::NotificationMode::off), MailishMessageInfoOpt());
        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleStore(std::move(err), std::move(data), "StoreMessage");
        };
        transaction_->fetch(q, h);
    }

    void getStoredMessage() {
        using namespace macs::pg::query;
        const auto q = query<MailboxEntriesByIds>( MailIdList( {mid_} ) );
        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleGetMessage(std::move(err), std::move(data), "MailboxEntriesByIds");
        };
        transaction_->fetch(q, h);
    }

    void storeDeletedMessage() {
        using namespace macs::pg::query;

        const auto q = queryUpdate<StoreDeletedMessage>(
                PGEnvelopeDeleted( envFactory_.releaseDeleted() ), mime_);
        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleStore(std::move(err), std::move(data), "StoreDeletedMessage");
        };
        transaction_->fetch(q, h);
    }

    void getDeletedMessage() {
        using namespace macs::pg::query;
        const auto q = query<DeletedMessagesByIds>( MailIdList( {mid_} ) );
        const auto h = [this, self=this->shared_from_this()](auto err, auto data){
            this->handleGetMessage(std::move(err), std::move(data), "DeletedMessagesByIds");
        };
        transaction_->fetch(q, h);
    }

    void handleDuplicates(error_code err, Mid mid) {
        if(err) {
            hook_(std::move(err));
        } else {
            mid_ = mid;
            (*this)();
        }
    }

    template <typename DataRange>
    void handleStore(error_code err, DataRange data, const std::string& query) {
        if (err) {
            hook_(std::move(err));
        } else if (data.empty()) {
            hook_(error_code(pgg::error::noDataReceived, "No data received as a result of " + query));
        } else {
            const auto v = pgg::cast< reflection::StoreMessage >( data.front() );
            mid_ = std::to_string( v.mid );
            (*this)();
        }
    }

    template <typename DataRange>
    void handleGetMessage(error_code err, DataRange data, const std::string& query) {
        if (err) {
            hook_(std::move(err));
        } else if (data.empty()) {
            hook_(error_code(pgg::error::noDataReceived, "No data received as a result of " + query));
        } else {
            auto v = pgg::cast< reflection::Envelope >( data.front() );
            storedMessage_ = makeEnvelope(labels_, std::move(v));
            (*this)();
        }
    }


    ConnProvider connProvider_;
    TransactionPtr transaction_;
    pgg::query::RepositoryPtr queryRepository_;
    std::string uid_;
    pgg::RequestInfo requestInfo_;

    macs::EnvelopeFactory envFactory_;
    MimeParts mime_;
    ThreadMeta meta_;
    LabelSet labels_;
    FolderSet folders_;
    bool ignoreDuplicates_;
    macs::EnvelopesRepository::NotificationMode notificationMode_;
    macs::EnvelopesRepository::StoreType storeType_;
    OnUpdateEnvelope hook_;
    std::string mid_;
    Envelope storedMessage_;
    pgg::Milliseconds timeout_;
    TabResolver tabResolver_;

    template <typename T, typename ...Args >
    T queryUpdate( Args&& ... args) const {
        return query<T>(requestInfo_, std::forward<Args>(args)...);
    }

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository_, uid_, std::forward<Args>(args)...);
    }
};

}   //namespace pg
}   //namespace macs

#include <boost/asio/unyield.hpp>
