#ifndef INCLUDE_INTERNAL_FOLDER_CLEAR_CASCADE_TRANSACTION_H
#define INCLUDE_INTERNAL_FOLDER_CLEAR_CASCADE_TRANSACTION_H

#include <internal/folder/repository.h>
#include <pgg/query/transactional.h>
#include <internal/folder/clear_cascade_data.h>
#include <pgg/range.h>
#include <internal/folder/query.h>
#include <pgg/cast.h>
#include <pgg/numeric_cast.h>
#include <internal/reflection/revision.h>
#include <internal/reflection/update_messages.h>

#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

class ClearCascadeTransaction {
    using Iter = ClearCascadeData::FoldersList::const_iterator;
public:
    ClearCascadeTransaction(pgg::query::RepositoryPtr qr, OnUpdateMessages h,
            const FolderSet &fs, const std::string& rootFid, const std::string& uid,
            const pgg::RequestInfo& requestInfo, pgg::Milliseconds timeout);
    template <typename Transactional>
    void operator()(pgg::query::Coroutine & coro, Transactional & t) {
        const auto onError = boost::bind(&ClearCascadeTransaction::checkError, this, _1);

        reenter( coro ) {
            yield t.begin( onError, timeout_ );

            for (folderIt = clearData.foldersToMove().begin();
                    folderIt != clearData.foldersToMove().end(); ++folderIt) {
                yield requestMoveFolder(t, *folderIt);
            }

            for (folderIt = clearData.foldersToDelete().begin();
                    folderIt != clearData.foldersToDelete().end(); ++folderIt) {
                yield requestMoveFolderMessages(t, *folderIt);
                yield requestDeleteFolder(t, *folderIt);
            }

            yield t.commit( onError );
            externalHook();
        }
    }

private:
    template <typename Transactional>
    void requestMoveFolder(Transactional& t, const Folder& folder) {
        using namespace macs::pg::query;
        const auto q = query<UpdateFolder>(
                FID(folder.fid()),
                FolderName(clearData.getNameForMove(folder)),
                ParentFolderId(clearData.parentFid()));
        t.fetch(q, [&](error_code e, auto data) {updateHandler(std::move(e), std::move(data));});
    }

    template <typename Transactional>
    void requestMoveFolderMessages(Transactional& t, const Folder& folder) {
        using namespace macs::pg::query;
        const auto q = query<MoveFolderMessages>(SrcFID(folder.fid()), DstFID(clearData.trashFid()));
        t.fetch(q, [&](error_code e, auto data) {moveMessagesHandler(std::move(e), std::move(data));});
    }

    template <typename Transactional>
    void requestDeleteFolder(Transactional& t, const Folder& folder) {
        using namespace macs::pg::query;
        const auto q = query<DeleteFolder>(FID(folder.fid()));
        t.fetch(q, [&](error_code e, auto data) {updateHandler(std::move(e), std::move(data));});
    }

    template <typename DataRange>
    void updateHandler(error_code result, DataRange data) {
        if (result) {
            hook(result);
        } else if (!data.empty()) {
            setRevision(static_cast<Revision>(pgg::cast<reflection::Revision>(data.front()).revision));
        }
    }

    template <typename DataRange>
    void moveMessagesHandler(error_code result, DataRange data) {
        if (result) {
            hook(result);
        } else if (!data.empty()) {
            const auto r = pgg::cast<reflection::UpdateMessages>(data.front());
            setRevision(PGG_NUMERIC_CAST(Revision, r.revision));
            affectedMessages += PGG_NUMERIC_CAST(std::size_t, r.affected);
        }
    }

    void checkError(error_code result) const;
    void externalHook() const;

    pgg::query::RepositoryPtr queryRepository;
    OnUpdateMessages hook;
    const std::string uid;

    ClearCascadeData clearData;
    Iter folderIt;

    Revision revision;
    std::size_t affectedMessages;
    const pgg::RequestInfo requestInfo;
    pgg::Milliseconds timeout_;

    void setRevision(Revision value) {
        if (value != NULL_REVISION) {
            revision = value;
        }
    }

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return queryRepository->query<T>(query::UserId(uid), requestInfo, std::forward<Args>(args)...);
    }
};

} //namespace pg
} //namespace macs

#include <boost/asio/unyield.hpp>

#endif //INCLUDE_INTERNAL_FOLDER_CLEAR_CASCADE_TRANSACTION_H
