#pragma once

#include <internal/query/comment.h>
#include <pgg/query/transactional.h>
#include <internal/folder/query.h>
#include <internal/folder/folders_converter.h>
#include <internal/hooks/wrap.h>

#include <macs_pg/logging.h>

#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

class CreateFolderByPath {
public:
    CreateFolderByPath(pgg::query::RepositoryPtr queryRepository, Folder::Path path, const Uid& uid,
            const pgg::RequestInfo& requestInfo, pgg::Milliseconds timeout, OnUpdateFolder hook);

    template <typename Transactional>
    void operator()(pgg::query::Coroutine& coroutine, Transactional& transactional) {
        const auto handler = [=] (error_code ec) {
            if (ec) {
                hook(std::move(ec));
            }
        };
        error_code ec;

        reenter(coroutine) {
            yield transactional.begin(handler, timeout);
            yield getFolders(transactional);
            currentName = path.begin();
            std::tie(currentName, parentFid) = advanceToFirstNonexistentFolder(currentName, std::move(parentFid));
            if (currentName != path.begin()) {
                lastFolder = folders.at(parentFid);
            }
            ec = canCreateFolders(currentName, path.end(), parentFid);
            if (ec) {
                hook(std::move(ec));
                yield break;
            }
            while (currentName != path.end()) {
                yield createFolder(transactional, *currentName, parentFid);
                parentFid = lastFolder.fid();
                ++currentName;
            }
            yield transactional.commit(handler);
            hook(std::move(lastFolder));
        }
    }

private:
    using PathIt = Folder::Path::const_iterator;

    pgg::query::RepositoryPtr queryRepository;
    Uid uid;
    Folder::Path path;
    pgg::RequestInfo requestInfo;
    pgg::Milliseconds timeout;
    OnUpdateFolder hook;

    PathIt currentName;
    FolderSet folders;
    Fid parentFid = Folder::noParent;
    Folder lastFolder;

    template <typename Transactional>
    void getFolders(Transactional& transactional) {
        using namespace macs::pg::query;
        transactional.fetch(
            query<AllFoldersList>(),
            [=] (auto ec, auto range) { this->onGetFolders(std::move(ec), std::move(range)); }
        );
    }

    template <class DataRange>
    void onGetFolders(error_code ec, DataRange range) {
        if (ec) {
            hook(std::move(ec));
        } else {
            this->folders = convertFolders(std::move(range));
        }
    }

    std::pair<PathIt, Fid> advanceToFirstNonexistentFolder(PathIt currentName, Fid parentFid) const;

    error_code canCreateFolders(PathIt begin, PathIt end, const Fid& parentFid) const;

    template <typename Transactional>
    void createFolder(Transactional& transactional, const std::string& name, const Fid& parentFid) {
        using namespace macs::pg::query;
        transactional.fetch(
            queryUpdate<CreateFolder>(FolderName(name), ParentFolderId(parentFid)),
            wrapHook(OnUpdateFolder([=] (auto ec, auto folder) { this->onCreateFolder(std::move(ec), std::move(folder)); }))
        );
    }

    void onCreateFolder(error_code ec, Folder folder);

    template <typename T, typename ... Args>
    T query(Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository, uid, std::forward<Args>(args) ...);
    }

    template <typename T, typename ... Args>
    T queryUpdate(Args&& ... args) const {
        return makeQueryWithComment<T>(*queryRepository, uid, requestInfo, std::forward<Args>(args) ...);
    }
};

#include <boost/asio/unyield.hpp>

} //namespace pg
} //namespace macs
