#ifndef INCLUDE_INTERNAL_FOLDER_MODIFY_MASTER_H
#define INCLUDE_INTERNAL_FOLDER_MODIFY_MASTER_H

#include <internal/folder/repository.h>
#include <pgg/query/transactional.h>
#include <pgg/range.h>
#include <internal/folder/query.h>
#include <boost/algorithm/string/split.hpp>

#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

class ModifyMaster {
public:
    using Path = std::list<std::string>;
    using Iter = Path::const_iterator;
    using Range = boost::iterator_range<Iter>;
    static constexpr auto root = "0";

    ModifyMaster(pgg::query::RepositoryPtr qr, OnUpdateFolder h,
            const Folder& f, Folder::Path fodlerPath, FolderSet fs, const std::string& uid,
            const pgg::RequestInfo& requestInfo, pgg::Milliseconds timeout);

    template<typename Transactional>
    void operator()(pgg::query::Coroutine & coro, Transactional & t) {
        using namespace macs::pg::query;
        reenter( coro ) {
            yield {
                auto onError = boost::bind(&ModifyMaster::checkError, this, _1);
                t.begin( onError, timeout_ );
            }
            while ( !parentsVec.empty() ) {
                if (folderSet.exists(parentsVec.front(), currentParent)) {
                    newParentFid = folderSet.fid(parentsVec.front(), currentParent);
                    if (newParentFid == folder.fid()) {
                        yield {
                            const std::string tempName = getUniqueFolderName();
                            auto q = query<UpdateFolder>(FolderId(folder.fid()), FolderName(tempName),
                                    ParentFolderId(Folder::noParent));
                            t.fetch( q, [&](error_code e, auto data) {updateHandler(std::move(e), std::move(data));});
                        }
                        yield {
                            auto q = query<CreateFolder>(FolderName(parentsVec.front()),
                                    ParentFolderId(currentParent));
                            t.fetch( q, [&](error_code e, auto data) {createHandler(std::move(e), std::move(data));});
                        }
                    } else {
                        currentParent = newParentFid;
                        parentsVec.advance_begin(1);
                    }
                } else yield {
                    auto q = query<CreateFolder>(FolderName(parentsVec.front()),
                            ParentFolderId(currentParent));
                    t.fetch( q, [&](error_code e, auto data) {createHandler(std::move(e), std::move(data));});
                }
            }
            yield {
                auto q = query<UpdateFolder>(FolderId(folder.fid()), FolderName(path.back()),
                        ParentFolderId(currentParent));
                t.fetch( q, [&](error_code e, auto data) {updateHandler(std::move(e), std::move(data));});
            }
            yield {
                auto onError = boost::bind(&ModifyMaster::checkError, this, _1);
                t.commit( onError );
            }
            externalHook();
        }
    }

    void checkError(error_code result);
private:

    template <typename DataRange>
    void createHandler(const error_code& result, DataRange data) {
        if (result) {
            hook(result, macs::Folder());
        } else if (!data.empty()) {
            const auto created = FolderFactory().fromReflection(pgg::cast<reflection::Folder>(data.front())).product();
            parentsVec.advance_begin(1);
            currentParent = created.fid();
        }
    }

    template <typename DataRange>
    void updateHandler(const error_code& result, DataRange data) {
        if (result) {
            hook(result, Folder());
        } else if (!data.empty()) {
            resultFolder = FolderFactory().fromReflection(pgg::cast<reflection::Folder>(data.front())).product();
        }
    }

    std::string getUniqueFolderName() const;

    void externalHook() const;

    pgg::query::RepositoryPtr queryRepository;
    const FolderSet folderSet;
    const std::string uid;
    std::string currentParent;
    const Folder folder;
    Folder resultFolder;
    OnUpdateFolder hook;
    const pgg::RequestInfo requestInfo;
    pgg::Milliseconds timeout_;

    Path path;
    Range parentsVec;
    std::string newParentFid;

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return queryRepository->query<T>(query::UserId(uid), requestInfo, std::forward<Args>(args)...);
    }
};

} //namespace pg
} //namespace macs

#include <boost/asio/unyield.hpp>

#endif //INCLUDE_INTERNAL_FOLDER_MODIFY_MASTER_H
