#pragma once

#include <macs/hooks.h>
#include <macs/folders_repository.h>
#include <macs/errors.h>

#include <pgg/database/database.h>
#include <internal/not_implemented.h>
#include <pgg/query/repository.h>
#include <internal/query/ids.h>
#include <internal/query/comment.h>
#include <macs_pg/logging.h>

namespace macs {
namespace pg {

typedef query::UserId UID;
typedef query::FolderId FID;
typedef query::Position POS;
typedef query::SrcFolderId SrcFID;
typedef query::DstFolderId DstFID;

template <typename DatabaseGenerator>
class FoldersRepository : public macs::FoldersRepository {
public:
    typedef macs::FoldersRepository Base;

    FoldersRepository(UserJournalPtr journal, const DatabaseGenerator& generator,
                      const std::string & uid, pgg::query::RepositoryPtr queryRepository,
                      const pgg::RequestInfo& requestInfo, pgg::Milliseconds transactionTimeout, logging::v2::LogPtr logger)
    : Base(journal), uid_(uid), queryRepository_(queryRepository), db(generator),
      requestInfo_(requestInfo), transactionTimeout_(transactionTimeout), logger_(logger)
    {}

    virtual ~FoldersRepository() {}

private:

    // macs methods
    virtual MailboxSpaceInfo syncGetMailboxSpaceInfo() const override NOT_IMPLEMENTED

    virtual void syncGetMailboxRevision(OnRevisionReceive hook) const override;

    virtual void syncGetFolders(OnFolders) const override;

    virtual void syncGetPop3Folders(OnPop3Folders) const override NOT_IMPLEMENTED

    // should we use entry instead of fid here?
    virtual void syncResetUnvisited(const string & fid, OnUpdate hook) const override;

    virtual void syncSubscribeToSharedFolders(
            const std::string& /*sharedFoldersSuid*/, OnFoldersMap) const override NOT_IMPLEMENTED
    virtual void syncUnsubscribeFromSharedFolders(
            const std::string& /*sharedFoldersSuid*/, OnFoldersMap) const override NOT_IMPLEMENTED

    virtual void syncCreateFolder(const std::string& name,
                                  const std::string& parent,
                                  const Folder::Symbol& symbol,
                                  OnUpdateFolder hook) const override;
    void syncGetOrCreateFolder(const std::string& name,
                               const std::string& parent,
                               const Folder::Symbol& symbol,
                               OnUpdateFolder hook) const override;
    virtual void syncCreateFolderByPath(Folder::Path path, OnUpdateFolder hook) const override;
    virtual void syncModifyFolder(const Folder& folder, OnUpdateFolder handler) const override;
    virtual void syncModifyFolderToPath(const Folder& folder, const Folder::Path& path,
                                        OnUpdateFolder handler) const override;
    virtual void syncSetPosition(const std::string& fid, size_t pos, OnUpdate hook) const override;
    virtual void syncEraseFolder(const std::string& fid, OnUpdate handler) const override;
    virtual void syncClearFolderCascade(const std::string& fid,
                                        const FolderSet & folders,
                                        OnUpdateMessages handler) const override;

    virtual void syncMoveAll(const Folder & src,
                             const Folder & dst,
                             OnUpdateMessages handler) const override;

    virtual void syncSetFolderSymbol(const std::string & fid, const Folder::Symbol & symbol,
            OnUpdate handler) const override;

    virtual void syncSetPop3(const std::vector<std::string> & fids,
                             const FolderSet & fs,
                             OnUpdate handler) const override;

    virtual void asyncSetArchivationRule(const Fid &fid,
                                         const Folder::ArchivationType &type,
                                         uint32_t keep_days,
                                         uint32_t max_size,
                                         OnUpdate handler) const override;

    virtual void asyncRemoveArchivationRule(const Fid &fid,
                                            OnUpdate handler) const override;

private:
    const std::string & uid() const {
        return uid_;
    }
    const pgg::query::Repository & queryRepository() const {
        return *queryRepository_;
    }
    template <typename T, typename ...ArgsT >
    T query(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                std::forward<ArgsT>(args) ...);
    }

    template <typename T, typename ...ArgsT >
    T queryUpdate(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                requestInfo_, std::forward<ArgsT>(args) ...);
    }
    std::shared_ptr<const FoldersRepository> getSelf() const {
        return std::dynamic_pointer_cast<const FoldersRepository>(pgg::share(this));
    }

    const std::string uid_;
    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
    const pgg::RequestInfo requestInfo_;
    pgg::Milliseconds transactionTimeout_;
    logging::v2::LogPtr logger_;
};

template <typename DatabaseGenerator>
FoldersRepositoryPtr createFoldersRepository(
    macs::UserJournalPtr journal,
    DatabaseGenerator dbg,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    const pgg::RequestInfo& requestInfo,
    pgg::Milliseconds timeout,
    logging::v2::LogPtr logger);

} // namespace pg
} // namespace macs

#include "repository.ipp"
