#include <internal/folder/create_folder_by_path.h>

#include <internal/folder/repository.h>

#include <internal/folder/query.h>
#include <pgg/cast.h>
#include <internal/folder/folders_converter.h>
#include <internal/folder/modify_master.h>
#include <internal/folder/clear_cascade_transaction.h>

#include <internal/hooks/wrap.h>
#include <internal/reflection/revision.h>
#include <internal/reflection/folder.h>

namespace macs {
namespace pg {

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncGetMailboxRevision(OnRevisionReceive hook) const {
    using R = reflection::Revision;
    const auto q = query<query::MailboxRevision>();
    db()->fetch(q, wrapHook<R>(std::move(hook), [] (R v) { return Revision(v.revision); },
            [=](std::ostream & s) { s << "syncGetMailboxRevision()";} ));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncGetFolders(OnFolders h) const {
    db()->fetch(query<query::AllFoldersList>(), wrapException(std::move(h), [log = logger_] (auto data) {
            std::vector<Folder> fs;
            boost::transform(data, std::back_inserter(fs), [] (auto row) {
                return FolderFactory().fromReflection(pgg::cast<reflection::Folder>(row)).product();
            });
            return normalize(std::move(fs), log, "FoldersRepository::syncGetFolders");
        }
    ));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncCreateFolder(const std::string& name,
        const std::string& parent, const Folder::Symbol& symbol, OnUpdateFolder hook) const {
    using query::FolderName;
    using query::ParentFolderId;

    const auto q = queryUpdate<query::CreateFolder>(FolderName(name), ParentFolderId(parent), symbol);

    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncGetOrCreateFolder(const std::string& name,
                                              const std::string& parent,
                                              const Folder::Symbol& symbol,
                                              OnUpdateFolder hook) const {
    using query::FolderName;
    using query::ParentFolderId;

    const auto q = queryUpdate<query::GetOrCreateFolder>(FolderName(name), ParentFolderId(parent), symbol);
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncCreateFolderByPath(Folder::Path path, OnUpdateFolder hook) const {
    pgg::query::fallback::runTransactional(db, queryRepository_,
        CreateFolderByPath(queryRepository_, std::move(path), uid(), requestInfo_,
                           transactionTimeout_, std::move(hook))
    );
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncEraseFolder(const std::string& fid, OnUpdate hook) const {
    const auto q = queryUpdate<query::DeleteFolder>(FID(fid));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncResetUnvisited(const string& fid, OnUpdate hook) const {
    const auto q = queryUpdate<query::ResetFolderUnvisited>(FID(fid));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncMoveAll(const Folder & src,
                                    const Folder & dst,
                                    OnUpdateMessages hook) const {
    const auto q = queryUpdate<query::MoveFolderMessages>(SrcFID(src.fid()), DstFID(dst.fid()));
    db()->fetch(q, wrapHook(std::move(hook)));
}


template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncClearFolderCascade(const std::string& fid,
                                               const FolderSet & folders,
                                               OnUpdateMessages hook) const {
    auto transaction = ClearCascadeTransaction(queryRepository_, std::move(hook), folders, fid, uid(),
            requestInfo_, transactionTimeout_);
    pgg::query::fallback::runTransactional(db, queryRepository_, std::move(transaction));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncModifyFolder(const Folder& folder, OnUpdateFolder hook) const {
    using namespace query;
    const auto q = queryUpdate<UpdateFolder>(
            FolderId(folder.fid()),
            FolderName(getFolderDisplayName(folder.name())),
            ParentFolderId(folder.parentId()));
    db()->fetch(q, wrapHook(std::move(hook)));
}


template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncModifyFolderToPath(const Folder& folder, const Folder::Path& path,
                                               OnUpdateFolder hook) const {
    getAllFolders([self = getSelf(), hook = std::move(hook), folder, path]
                   (error_code e, FolderSet folders) mutable {
        if(e) {
            hook(e);
            return;
        }

        auto master = ModifyMaster(self->queryRepository_, std::move(hook),
                folder, path, folders, self->uid(), self->requestInfo_, self->transactionTimeout_);
        pgg::query::fallback::runTransactional(self->db, self->queryRepository_,
                std::move(master));
    });
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncSetFolderSymbol(const std::string & fid,
            const Folder::Symbol & symbol, OnUpdate hook) const {
    const auto q = queryUpdate<query::UpdateFolderSymbol>(FID(fid), symbol);
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncSetPosition(const std::string& fid, size_t pos,
        OnUpdate hook) const {
    const auto q = query<query::SetFolderPosition>(FID(fid), POS(pos));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::syncSetPop3(const std::vector<std::string> & fidsOn,
                                    const FolderSet & fs,
                                    OnUpdate hook) const {
    std::vector<std::string> fidsOff;
    using namespace boost::adaptors;
    boost::copy(fs | map_keys | filtered(
            [&fidsOn](const std::string& fid){ return boost::find(fidsOn, fid) == fidsOn.end(); }),
        std::back_inserter(fidsOff));

    const auto q = queryUpdate<query::SetPop3>(query::FolderIdListOn(fidsOn), query::FolderIdListOff(fidsOff));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::asyncSetArchivationRule(const Fid &fid,
                                                const Folder::ArchivationType &type,
                                                uint32_t keep_days,
                                                uint32_t max_size,
                                                OnUpdate hook) const {
    const auto q = queryUpdate<query::SetFolderArchivationRules>(
                query::FolderId(fid),
                query::ArchivationType(type),
                query::ArchivationTtl(keep_days),
                query::MaxFolderSize(max_size));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void FoldersRepository<DatabaseGenerator>::asyncRemoveArchivationRule(const Fid &fid,
                                                   OnUpdate hook) const {
    const auto q = queryUpdate<query::RemoveFolderArchivationRules>(
                query::FolderId(fid));
    db()->fetch(q, wrapHook(std::move(hook)));
}

}
}
