#pragma once

#include <pgg/enumeration.h>
#include <internal/folder/folder_type.h>
#include <boost/bimap.hpp>
#include <macs/folder.h>

namespace macs::pg::symbols {

class FolderTypeConverter {
public:
    FolderType fromSymbol( const macs::Folder::Symbol & symbol ) const {
        return get(map().right, symbol, FolderType::unknown);
    }
    const macs::Folder::Symbol & toSymbol(FolderType type) const {
        return get(map().left, type, Sym::none );
    }
private:
    using Sym = macs::Folder::Symbol;
    using Map = boost::bimap<FolderType,Sym>;

    static Map makeMap() {
        using V = Map::value_type;
        Map m;
        m.insert(V(FolderType::user, Folder::Symbol::none));
        m.insert(V(FolderType::inbox, Folder::Symbol::inbox));
        m.insert(V(FolderType::sent, Folder::Symbol::sent));
        m.insert(V(FolderType::trash, Folder::Symbol::trash));
        m.insert(V(FolderType::spam, Folder::Symbol::spam));
        m.insert(V(FolderType::drafts, Folder::Symbol::drafts));
        m.insert(V(FolderType::outbox, Folder::Symbol::outbox));
        m.insert(V(FolderType::archive, Folder::Symbol::archive));
        m.insert(V(FolderType::templates, Folder::Symbol::template_));
        m.insert(V(FolderType::pending, Folder::Symbol::pending));
        m.insert(V(FolderType::discount, Folder::Symbol::discount));
        m.insert(V(FolderType::zombie, Folder::Symbol::zombie_folder));
        m.insert(V(FolderType::unsubscribe, Folder::Symbol::unsubscribe));
        m.insert(V(FolderType::hidden_trash, Folder::Symbol::hidden_trash));
        m.insert(V(FolderType::restored, Folder::Symbol::restored));
        m.insert(V(FolderType::reply_later, Folder::Symbol::reply_later));
        return m;
    }

    static const Map & map() {
        static const Map m = makeMap();
        return m;
    }

    template <typename T>
    using CRef = const typename T::value_type::second_type &;

    template <typename Map, typename Key>
    CRef<Map> get( const Map & map, const Key & key, CRef<Map> defVal ) const {
        const auto i = map.find(key);
        return  i==map.end() ? defVal : i->second;
    }
};


inline Folder::Symbol defineFolderSymbol(macs::pg::FolderType const & folderType) {
    return FolderTypeConverter().toSymbol(folderType);
}

} // namespace macs::pg::symbols
