#pragma once

#include <internal/reflection/attaches_counters.h>
#include <internal/reflection/revision.h>
#include <internal/reflection/update_messages.h>
#include <internal/reflection/label.h>
#include <internal/reflection/folder.h>
#include <internal/reflection/mid.h>
#include <internal/reflection/find_threads.h>
#include <internal/reflection/count.h>
#include <internal/reflection/stid_by_mid.h>
#include <internal/reflection/imap_id_result.h>
#include <internal/reflection/tab.h>
#include <internal/reflection/mime_and_attach_by_mid.h>
#include <internal/label/factory.h>
#include <internal/folder/factory.h>
#include <internal/tab/factory.h>
#include <internal/envelope/mime_parts.h>
#include <macs_pg/error.h>
#include <pgg/cast.h>
#include <pgg/hooks/wrap.h>
#include <pgg/numeric_cast.h>
#include <macs/hooks.h>
#include <macs/types.h>
#include <memory>

namespace macs {
namespace pg {

template <class Reflection, class Hook, class GetValue, typename ... Args>
auto wrapHook(Hook hook, GetValue get, Args&& ... args) {
    return pgg::wrapHook<Reflection>(std::move(hook), std::move(get), std::forward<Args>(args)...);
}

inline auto wrapHook(OnUpdate hook) {
    return wrapHook<reflection::Revision>(std::move(hook), [](auto v) {
        return PGG_NUMERIC_CAST(Revision, v.revision);
    });
}

inline auto wrapHook(OnUpdateMessages hook) {
    return wrapHook<reflection::UpdateMessages>(std::move(hook),[](auto v) {
        return UpdateMessagesResult{
            PGG_NUMERIC_CAST(Revision, v.revision),
            PGG_NUMERIC_CAST(std::size_t, v.affected)
        };
    });
}

inline auto wrapHook(OnUpdateLabel hook) {
    return wrapHook<reflection::Label>(std::move(hook),[](auto v) {
        return macs::pg::LabelFactory().fromReflection(v).product();
    });
}

inline auto wrapHook(OnUpdateFolder hook) {
    return wrapHook<reflection::Folder>(std::move(hook),[](auto v) {
        return macs::pg::FolderFactory().fromReflection(v).product();
    });
}

inline auto wrapHook(OnMidsReceive hook) {
    return wrapHook<reflection::Mid>(std::move(hook),[](auto v) {
        return std::to_string(v.mid);
    });
}

inline auto wrapHook(OnThreadIdsReceive hook) {
    return wrapHook<reflection::FindThreads>(std::move(hook),[](auto v) {
        return std::to_string(v.tid);
    });
}

inline auto wrapHook(OnMidsWithStidsReceive hook) {
    return wrapHook<reflection::StidByMid>(std::move(hook), [](auto v) {
        return MidStid(std::to_string(v.mid), v.st_id);
    });
}

template <typename... Args>
auto wrapHook(OnCountReceive hook, Args&&... args) {
    return wrapHook<reflection::Count>(
        std::move(hook),
        [](auto v) { return PGG_NUMERIC_CAST(std::size_t, v.count); },
        std::forward<Args>(args)...
    );
}

inline auto wrapHook(OnImapId hook) {
    return wrapHook<reflection::ImapIdResult>(std::move(hook), [](auto v) {
        return ImapIdResult{
            PGG_NUMERIC_CAST(Revision, v.revision),
            PGG_NUMERIC_CAST(uint64_t, v.imap_id)
        };
    });
}

inline auto wrapHook(OnAttachesCounters hook) {
    return wrapHook<reflection::AttachesCountersResult>(std::move(hook), [](auto v) {
        return AttachesCounters{
            PGG_NUMERIC_CAST(uint64_t, v.has_attaches_count),
            PGG_NUMERIC_CAST(uint64_t, v.has_attaches_seen)
        };
    });
}

inline auto wrapHook(OnMidsWithMimesAndAttaches hook) {
    return wrapHook<reflection::MimeAndAttachesByMid>(std::move(hook), [](auto v) {
        std::vector<AttachmentDescriptor> attaches;
        attaches.reserve(v.attaches.size());

        for (auto &attach : v.attaches) {
            attaches.emplace_back(AttachmentDescriptor{
                    std::move(attach.hid)
                    , std::move(attach.type)
                    , std::move(attach.filename)
                    , std::move(PGG_NUMERIC_CAST(std::size_t, attach.size))
            });
        }

        return std::make_tuple(std::to_string(v.mid), std::move(v.st_id),
                               v.mime ? toMime(std::move(*v.mime)) : MimeParts(),
                               std::move(attaches));
    });
}

template <typename Hook, typename GetValue>
auto wrapException(Hook hook, GetValue get) {
    return [hook = std::move(hook), get = std::move(get)] (error_code e, auto data) { 
        if (e) {
            hook(std::move(e));
            return;
        }
        try {
            hook(get(std::move(data)));
        } catch (const boost::coroutines::detail::forced_unwind&) {
            throw;
        } catch (const boost::system::system_error& e) {
            hook(error_code{e.code(), e.what()});
        } catch (const std::exception& e) {
            hook(error_code{error::exceptionInConverter, e.what()});
        }
    };
}

} // namespace pg
} // namespace macs
