#pragma once

#include <macs/envelopes_repository.h>
#include <macs/imap_repository.h>
#include <internal/not_implemented.h>
#include <internal/imap/query_imap.h>
#include <pgg/query/repository.h>
#include <internal/query/ids.h>
#include <pgg/query/ranges.h>
#include <internal/query/comment.h>

namespace macs {
namespace pg {

typedef query::FolderId FID;
typedef query::LabelId LID;
typedef query::UserId UID;

/// Envelopes access and manipulation class for IMAP
template <typename DatabaseGenerator>
class ImapRepository : public macs::ImapRepository {
public:
    typedef macs::ImapRepository Base;
    ImapRepository(UserJournalPtr journal, const DatabaseGenerator& generator,
            const std::string & uid, pgg::query::RepositoryPtr queryRepository, const pgg::RequestInfo& requestInfo)
        : Base(journal)
        , uid_(uid)
        , queryRepository_(queryRepository)
        , db(generator)
        , requestInfo_(requestInfo)
    {}

    virtual ~ImapRepository() {}

protected:
    // Get mapping from uids to imap numbers
    virtual void asyncImapGetUidMap(
        const ImapFolder& folder,
        imap_uid_map_entry handler) const override;

    /// load base imap message list
    virtual void asyncImapGetMails(const ImapFolder& folder,
        uint64_t low, uint64_t hi, ImapListMode mode,
        imap_envelope_entry handler) const override;

    /// load base imap message list
    virtual void asyncImapGetMailsChunk(const ImapFolder& folder,
        uint64_t low, uint64_t hi, uint64_t maxChunkSize,
        imap_envelope_entry handler) const override;

    /// load changed messages in folder
    virtual void asyncImapGetChanges(const ImapFolder& folder,
        imap_envelope_changes_entry handler) const override;

    /// load messages by imap message_id
    virtual void asyncImapGetByMessageId(
        const ImapFolder& folder, const std::string& messageId,
        imap_envelope_details_entry handler) const override;

    /// load imap messages details
    virtual void asyncImapGetDetails(
        const ImapFolder& /*folder*/, uint64_t /*low*/, uint64_t /*hi*/,
        imap_envelope_details_entry /*handler*/) const override NOT_IMPLEMENTED

    /// load imap messages details
    virtual void asyncImapGetDetails(
        const ImapFolder& folder, const std::vector<uint64_t>& uids,
        imap_envelope_details_entry handler) const override;

    /// load imap deleted messages
    virtual void asyncImapGetDeleted(
        const ImapFolder& folder,
        imap_envelope_entry handler) const override;

    virtual void asyncImapUpdateFlags(
        const ImapFolder& /*folder*/, const LabelSet& /*del*/,
        const LabelSet& /*add*/, std::vector<uint64_t> /*uid*/,
        imap_envelope_oper /*handler*/) const override NOT_IMPLEMENTED

    virtual void asyncImapDelEnvelope(
        const ImapFolder& /*folder*/, const std::vector<uint64_t>& /*uids*/,
        imap_envelope_oper /*handler*/) const override NOT_IMPLEMENTED

    virtual void asyncImapCopyEnvelope(
        const ImapFolder& /*from*/, const ImapFolder& /*to*/,
        const std::vector<uint64_t>& /*uids*/,
        imap_envelope_oper /*handler*/) const override NOT_IMPLEMENTED

    virtual void asyncImapMoveEnvelope(
        const ImapFolder& /*from*/, const ImapFolder& /*to*/,
        const std::vector<uint64_t>& /*uids*/,
        imap_envelope_oper /*handler*/) const override NOT_IMPLEMENTED

    virtual void asyncImapGetUnsubscribed(OnImapUnsubscribed) const override;

    virtual void asyncImapSubscribeFolder(
        const std::vector<std::string>& fullName,
        OnUpdate handler) const override;

    virtual void asyncImapUnsubscribeFolder(
        const std::vector<std::string>& fullName,
        OnUpdate handler) const override;

    virtual void asyncImapRegenerateImapId(
        const std::string& mid, OnImapId handler) const override;

    virtual void asyncPop3GetFolders(
        const std::string& /*suid*/,
        pop3_folder_entry /*handler*/) const override NOT_IMPLEMENTED

    virtual void asyncPop3LoadFullMessagesSingleFolder(
        const Fid& /*folderId*/,
        const Fid&/*spamFid*/,
        pop3_full_message_entry /*handler*/) const override NOT_IMPLEMENTED

    virtual void asyncPop3LoadFullMessages(
        const std::string& /*suid*/,
        const Fid& /*spamFid*/,
        const FidList& /*fids*/,
        pop3_full_message_entry /*handler*/) const override;

    virtual void asyncPop3DeleteMessages(
        const std::list<std::string>& mids,
        OnUpdate handler) const override;


private:
    const std::string & uid() const { return uid_; }

    const pgg::query::Repository & queryRepository() const {
        return *queryRepository_;
    }
    template <typename T, typename ...ArgsT >
    T query(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                std::forward<ArgsT>(args) ...);
    }

    template <typename T, typename ...ArgsT >
    T queryUpdate(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                pgg::RequestInfo(requestInfo_), std::forward<ArgsT>(args) ...);
    }

    const std::string uid_;
    pgg::query::RepositoryPtr queryRepository_;
    const DatabaseGenerator db;
    const pgg::RequestInfo requestInfo_;
};

template <typename DatabaseGenerator>
ImapRepositoryPtr createImapRepository(
    macs::UserJournalPtr journal,
    DatabaseGenerator dbg,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    const pgg::RequestInfo& requestInfo);

} // namespace pg
} // namespace macs

#include "repository_imap.ipp"
