#include <internal/imap/repository_imap.h>
#include <internal/imap/query_imap.h>
#include <internal/imap/convert_imap.h>
#include <boost/variant/variant.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <pgg/error.h>
#include <pgg/range.h>
#include <pgg/cast.h>
#include <internal/hooks/wrap.h>

namespace macs {
namespace pg {

namespace detail {

template<typename Handler, class Result, class Factory, class Reflection>
struct HandleImapQuery {
    HandleImapQuery(Handler handler) : handler(std::move(handler)) {}

    template <typename DataRange>
    void operator()(error_code e, DataRange data) const {
        if (e) {
            handler(e, Result());
            return;
        }

        Result res;
        boost::transform(data, std::back_inserter(res), [](auto row) {
            return Factory().fromReflection(pgg::cast<Reflection>(row)).product();
        });
        handler(e, std::move(res));
    }

    Handler handler;
};

typedef HandleImapQuery<
    macs::imap_uid_map_entry,
    ImapUidMapChunk,
    ImapUidMapFactory, reflection::ImapUidMap > HandleImapUidMapQuery;

typedef HandleImapQuery<
    macs::imap_envelope_entry,
    ImapEnvelopeChunk,
    ImapEnvelopeFactory, reflection::ImapEnvelope > HandleImapGetMailsQuery;

typedef HandleImapQuery<
    macs::imap_envelope_changes_entry,
    ImapEnvelopeChangesChunk,
    ImapEnvelopeChangesFactory, reflection::ImapEnvelopeChanges > HandleImapChangesQuery;

typedef HandleImapQuery<
    macs::imap_envelope_details_entry,
    ImapEnvelopeDetailsChunk,
    ImapEnvelopeDetailsFactory, reflection::ImapEnvelopeDetails > HandleImapEnvelopeDetailsQuery;

typedef HandleImapQuery<
    macs::imap_envelope_details_entry,
    ImapEnvelopeDetailsChunk,
    ImapEnvelopeByMessageIDFactory, reflection::ImapEnvelope > HandleImapEnvelopeByMessageIDQuery;

typedef HandleImapQuery<
    macs::OnImapUnsubscribed,
    ImapUnsubscribedDataChunk,
    ImapUnsubscribedDataFactory, reflection::ImapUnsubscribedData > HandleImapUnsubscribedDataQuery;

} // namespace detail


template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetUidMap(
    const ImapFolder& folder,
    imap_uid_map_entry handler) const
{
    detail::HandleImapUidMapQuery wrapper(std::move(handler));
    db()->fetch( query<query::ImapGetUidMap>(FID(folder.fid)), wrapper);
}


template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetMails(const ImapFolder& folder,
    uint64_t low, uint64_t hi, ImapListMode /*mode*/,
    imap_envelope_entry handler) const
{
    detail::HandleImapGetMailsQuery wrapper(std::move(handler));
    db()->fetch(
        query<query::ImapGetMails>(FID(folder.fid), pgg::query::ImapIdRange(low, hi)),
        wrapper);
}


template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetMailsChunk(const ImapFolder& folder,
    uint64_t low, uint64_t hi, uint64_t maxChunkSize,
    imap_envelope_entry handler) const
{
    detail::HandleImapGetMailsQuery wrapper(std::move(handler));
    db()->fetch(
        query<query::ImapGetMailsChunk>(FID(folder.fid), pgg::query::ImapIdRange(low, hi), maxChunkSize),
        wrapper);
}


template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetChanges(const ImapFolder& folder,
        imap_envelope_changes_entry handler) const
{
    detail::HandleImapChangesQuery wrapper(std::move(handler));
    db()->fetch(
        query<query::ImapGetChanges>(FID(folder.fid), query::ImapRevision(folder.revision)),
        wrapper);
}


template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetDetails(
    const ImapFolder&, const std::vector<uint64_t>& mids,
    imap_envelope_details_entry handler) const
{
    detail::HandleImapEnvelopeDetailsQuery wrapper(std::move(handler));
    auto m = mids | boost::adaptors::transformed(&boost::lexical_cast<std::string, uint64_t>);
    db()->fetch(query<query::ImapGetDetails>(query::MailIdList({m.begin(), m.end()})), wrapper);
}

template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetDeleted(
    const ImapFolder& folder,
    imap_envelope_entry handler) const
{
    detail::HandleImapGetMailsQuery wrapper(std::move(handler));
    db()->fetch(
        query<query::ImapGetDeleted>(FID(folder.fid)),
        wrapper);
}

template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetByMessageId(
    const ImapFolder& folder, const std::string& messageId,
    imap_envelope_details_entry handler) const
{
    detail::HandleImapEnvelopeByMessageIDQuery wrapper(std::move(handler));
    db()->fetch(
        query<query::ImapGetMailsByMessageId>(FID(folder.fid), query::ImapMessageId(messageId)),
        wrapper);
}

//-----------------------------------------------------------------------------
// IMAP subscribe/unsubscribe support

template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapGetUnsubscribed(OnImapUnsubscribed handler) const
{
    detail::HandleImapUnsubscribedDataQuery wrapper(std::move(handler));
    db()->fetch(query<query::ImapGetUnsubscribed>(), wrapper);
}

template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapSubscribeFolder(
    const std::vector<std::string>& fullNameParts,
    OnUpdate hook) const
{
    auto subscribeQuery = query<query::ImapSubscribe>(query::ImapFullName(fullNameParts));
    db()->fetch(subscribeQuery, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapUnsubscribeFolder(
    const std::vector<std::string>& fullNameParts,
    OnUpdate hook) const
{
    auto unsubscribeQuery = query<query::ImapUnsubscribe>(query::ImapFullName(fullNameParts));
    db()->fetch(unsubscribeQuery, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncImapRegenerateImapId(
    const std::string& mid,
    OnImapId hook) const
{
    auto regenerateImapId = query<query::ImapRegenerateImapId>(query::MailId(mid));
    db()->fetch(regenerateImapId, wrapHook(std::move(hook)));
}


//-----------------------------------------------------------------------------
// POP3 Implementation

namespace detail {

using HandlePop3Messages = HandleImapQuery<
    macs::pop3_full_message_entry,
    Pop3MessageChunk,
    Pop3MessageFactory, reflection::Pop3Message >;

} // namespace detail

template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncPop3LoadFullMessages(
    const std::string& /*suid*/,
    const Fid& /*spamFid*/,
    const FidList& fids,
    pop3_full_message_entry handler) const
{
    detail::HandlePop3Messages wrapper(std::move(handler));
    db()->fetch(query<query::Pop3GetMessages>(query::FolderIdList(fids)), wrapper);
}


template <typename DatabaseGenerator>
void ImapRepository<DatabaseGenerator>::asyncPop3DeleteMessages(
    const std::list<std::string>& mids,
    OnUpdate hook) const
{
    auto deleteQuery = query<query::Pop3DeleteMessages>(query::MailIdList(mids));
    db()->fetch(deleteQuery, wrapHook(std::move(hook)));
}


} // namespace pg
} // namespace macs
