#include <internal/label/repository.h>

#include <internal/label/query.h>
#include <internal/label/fake.h>
#include <internal/label/factory.h>

#include <pgg/range.h>
#include <internal/hooks/wrap.h>
#include <internal/reflection/count.h>
#include <internal/reflection/label.h>

namespace macs {
namespace pg {

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncGetLabels(OnLabels h) const {
    db()->fetch(query<query::AllLabelsList>(), wrapException(std::move(h), [](auto data) {
        std::vector<Label> labels;
        auto ins = std::back_inserter(labels);
        boost::copy(fakeLabels(), ins);
        boost::transform(data, ins, [](auto row) {
            return LabelFactory().fromReflection(pgg::cast<reflection::Label>(row)).product();
        });

        LabelSet res;
        for (auto& l : labels) {
            const auto lid = l.lid();
            if ( !res.insert({lid, std::move(l)}).second ) {
                throw system_error{error_code{macs::error::noSuchLabel, "Duplicate label id " + lid}};
            }
        }

        return res;
    })
    );
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncGetThreadsCount(const std::string& lid, OnCountReceive hook) const {
    auto handler = wrapHook(
        std::move(hook),
        [=](std::ostream & s) { s << "syncGetThreadsCount(" << lid << ")"; }
    );
    db()->fetch(query<query::ThreadsCountWithLabel>(query::LabelId(lid)), std::move(handler));
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncClearLabel(const Label& label, OnUpdateMessages hook) const {
    db()->fetch(queryUpdate<query::ClearLabel>(query::LabelId(label.lid())), wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncCreateLabel(const std::string& name, const std::string& color,
        const Label::Type& type, OnUpdateLabel hook) const {
    const query::LabelType labelType(LabelTypeConverter().fromMacsLabelType(type).toString());

    const auto q = queryUpdate<query::CreateLabel>(query::LabelName(name), labelType, query::LabelColor(color));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncCreateLabel(const Label::Symbol& symbol, OnUpdateLabel hook) const {
    const auto name = symbols::defineLabelName(symbol);
    if (name.empty()) {
        std::ostringstream stream;
        stream << __PRETTY_FUNCTION__ << ": creation of system label with symbol "
                << symbol.title() << " is forbidden.";
        hook(error_code(macs::error::invalidArgument, stream.str()));
        return;

    }
    return syncCreateLabel(name, "", Label::Type::system, std::move(hook));
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncEraseLabel(const std::string& lid, OnUpdate hook) const {
    db()->fetch(queryUpdate<query::EraseLabel>(query::LabelId(lid)), wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncModifyLabel(const Label& label, OnUpdateLabel hook) const {
    const auto q = queryUpdate<query::UpdateLabel>(query::LabelId(label.lid()),
                                             query::LabelName(label.name()),
                                             query::LabelColor(label.color()));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncGetOrCreateLabel(const std::string& name, const std::string& color,
        const Label::Type& type, OnUpdateLabel hook) const {
    const query::LabelType labelType(LabelTypeConverter().fromMacsLabelType(type).toString());

    const auto q = queryUpdate<query::ResolveLabels>(query::LabelName(name), labelType, query::LabelColor(color));
    db()->fetch(q, wrapHook(std::move(hook)));
}

template <typename DatabaseGenerator>
void LabelsRepository<DatabaseGenerator>::syncGetOrCreateLabel(const Label::Symbol& symbol, OnUpdateLabel hook) const {
    const auto fakes = fakeLabels();
    const auto found = fakes.find(symbol);
    if (found != fakes.end()) {
        hook(*found);
    } else {
        const auto name = symbols::defineLabelName(symbol);
        if (name.empty()) {
            std::ostringstream stream;
            stream << __PRETTY_FUNCTION__ << ": creation of system label with symbol "
                    << symbol.title() << " is forbidden.";
            hook(error_code(macs::error::invalidArgument, stream.str()));
            return;
        }
        syncGetOrCreateLabel(name, "", Label::Type::system, std::move(hook));
    }
}


} // namespace
} // namespace
