#ifndef INCLUDE_INTERNAL_LABEL_TABLE_H_161821042005
#define INCLUDE_INTERNAL_LABEL_TABLE_H_161821042005

#include <macs/label.h>

#include <boost/multi_index_container.hpp>
#include <boost/multi_index/hashed_index.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/mem_fun.hpp>
#include <boost/multi_index/tag.hpp>

namespace macs {
namespace pg {
namespace label {

namespace mi = boost::multi_index;

class Table {
public:
    struct IdTag {};
    struct SymbolTag {};
    using Symbol = Label::Symbol;

    struct SymbolHash {
        int operator () (const Symbol & s) const { return s.code(); }
    };

    using Item = Label;
    using Items = boost::multi_index_container<
        Item,
        mi::indexed_by<
            mi::hashed_unique<
                mi::tag<IdTag>,
                mi::const_mem_fun<Item, const std::string &, &Item::lid>
            >,
            mi::hashed_unique<
                mi::tag<SymbolTag>,
                mi::const_mem_fun<Item, const Symbol &, &Item::symbolicName>,
                SymbolHash
            >
        >
    >;

    using const_iterator = typename Items::const_iterator;
    using iterator = typename Items::iterator;

    Table(std::initializer_list<Label> list) : items(list) {}

    auto insert(Label label) -> decltype(Items().insert(std::move(label))) {
        return items.insert(std::move(label));
    }

    std::size_t size() const { return items.size(); }

    iterator begin() { return items.begin(); }
    iterator end() { return items.end(); }
    const_iterator begin() const { return items.begin(); }
    const_iterator end() const { return items.end(); }
    const_iterator cbegin() const { return begin(); }
    const_iterator cend() const { return end(); }

    const_iterator find(const std::string & lid) const { return find<IdTag>(this, lid); }
    iterator find(const std::string & lid) { return find<IdTag>(this, lid); }
    const_iterator find(const Symbol & s) const { return find<SymbolTag>(this, s); }
    iterator find(const Symbol & s) { return find<SymbolTag>(this, s); }

    const Item & at(const std::string & lid) const { return at<IdTag>(this, lid); }
    const Item & at(const Symbol & s) const { return at<SymbolTag>(this, s); }
private:
    template <typename Key, typename T, typename Self>
    static auto find(Self * self, const T & key) -> decltype(self->begin()) {
        auto & items = self->items;
        return items.template project<0>(items.template get<Key>().find(key));
    }
    template <typename Key, typename T, typename Self>
    static auto at(Self * self, const T & key) -> typename decltype(self->begin())::reference {
        const auto i = find<Key>(self, key);
        if(i==self->end()) {
            throw std::out_of_range("No such entry with key " + self->toString(key));
        }
        return *i;
    }
    std::string toString(const std::string & lid) const {
        return "lid:" + lid;
    }
    std::string toString(const Symbol & symbol) const {
        return "symbol:" + symbol.title();
    }
    Items items;
};

} // namespace label
} // namespace pg
} // namespace macs

#endif /* INCLUDE_INTERNAL_LABEL_TABLE_H_161821042005 */
