#pragma once

#include <pgg/query/transactional.h>
#include <internal/folder/query.h>
#include <internal/folder/factory.h>
#include <internal/mailish/query.h>
#include <internal/reflection/folder.h>

#include <macs/folders_repository.h>

#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

class CreateMailishFolder {
public:
    CreateMailishFolder(pgg::query::RepositoryPtr qr, FoldersRepositoryPtr foldersRepository, OnMailishFolder hook,
        const std::string& uid, const std::string& name, const std::string& parentId, const Folder::Symbol& symbol,
        const MailishFolderInfo& mailishInfo, pgg::Milliseconds timeout, FolderSet folders)
        : queryRepository(qr),
          foldersRepository(foldersRepository),
          folders(std::move(folders)),
          hook(std::move(hook)),
          uid(uid),
          name(name),
          parentId(parentId),
          symbol(symbol),
          mailishInfo(mailishInfo),
          transactionTimeout(timeout)
    {
    }

    template <typename Transactional>
    void operator()(pgg::query::Coroutine & coro, Transactional & t) {
        using namespace macs::pg::query;
        auto onError = boost::bind(&CreateMailishFolder::checkError, this, _1);
        reenter( coro ) {
            yield t.begin(onError, transactionTimeout);

            error = folders.checkCanCreateFolder(name, parentId, false);
            if (error) {
                return hook(error, MailishFolder());
            }

            yield {
                auto q = query<CreateFolder>(FolderName(name), ParentFolderId(parentId), symbol);
                t.fetch(q, [&](error_code e, auto data) {onCreate(std::move(e), std::move(data));});
            }

            if (error) {
                return hook(error, MailishFolder());
            }

            yield {
                auto q = query<MailishInitFolder>(FolderId(fid), mailishInfo);
                t.execute(q, onError);
            }
            yield t.commit(onError);
            foldersRepository->resetFoldersCache();
            {
                MailishFolderFactory factory;
                factory
                    .fid(fid)
                    .path(mailishInfo.externalPath())
                    .uidValidity(mailishInfo.uidValidity());
                hook(error_code(), factory.release());
            }
        }
    }

private:
    void checkError(error_code err) {
        if (err) {
            hook(err, MailishFolder());
        }
    }

    template <typename DataRange>
    void onCreate(error_code err, DataRange range) {
        if (err) {
            hook(err);
        } else if (!range.empty()) {
            auto folder = FolderFactory().fromReflection(pgg::cast<reflection::Folder>(range.front())).product();
            fid = std::move(folder.fid());
        } else {
            error = error_code(pgg::error::noDataReceived,
                "No data received as a result of CreateFolder");
        }
    }

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return queryRepository->query<T>(query::UserId(uid), std::forward<Args>(args)...);
    }

    pgg::query::RepositoryPtr queryRepository;
    FoldersRepositoryPtr foldersRepository;
    FolderSet folders;
    OnMailishFolder hook;
    std::string uid;
    std::string name;
    std::string parentId;
    Folder::Symbol symbol;
    MailishFolderInfo mailishInfo;
    std::string fid;
    pgg::Milliseconds transactionTimeout;
    error_code error;
};

} // namespace pg
} // namespace macs

#include <boost/asio/unyield.hpp>
