#pragma once

#include <pgg/query/transactional.h>
#include <internal/envelope/move_message.h>
#include <internal/mailish/query.h>

#include <boost/asio/yield.hpp>

namespace macs {
namespace pg {

template <typename ConnProvider, typename TransactionPtr>
class MoveMailishMessages: public boost::asio::coroutine,
                           public std::enable_shared_from_this<MoveMailishMessages<ConnProvider, TransactionPtr>> {
public:
    using Mids = std::list<std::string>;
    using ImapIds = std::vector<uint64_t>;

    MoveMailishMessages (ConnProvider conn,
                         TransactionPtr t,
                         pgg::query::RepositoryPtr qr,
                         pgg::RequestInfo ri,
                         OnExecute hook,
                         const std::string& uid,
                         const std::string& srcFid,
                         const std::string& dstFid,
                         const std::optional<Tab::Type>& dstTab,
                         const MailishMoveCoordsChunk& chunk,
                         const Fid& inboxFid,
                         const TabSet& tabs,
                         pgg::Milliseconds timeout)
        : connProvider(conn),
          transaction(t),
          queryRepository(qr),
          requestInfo(ri),
          hook(std::move(hook)),
          uid(uid),
          srcFid(srcFid),
          dstFid(dstFid),
          moveCoords(chunk),
          transactionTimeout(timeout)
    {
        MoveMessageArgs moveArgs{{}, dstFid, dstTab, inboxFid, tabs};
        for (auto& coords: chunk) {
            moveArgs.mids.push_back(coords.mid());
        }

        moveFunctor_ = [t, qr, uid, ri, moveArgs](OnUpdate hook) {
            ::macs::pg::moveMessage(t, qr, uid, ri, moveArgs, hook);
        };
    }

    void operator()() {
        using namespace macs::pg::query;
        auto cb = boost::bind(&MoveMailishMessages::checkError, this->shared_from_this(), _1);

        reenter (this) {
            yield transaction->begin(connProvider, std::move(cb), transactionTimeout);

            yield moveMailishMessages();

            yield moveMessages();

            yield transaction->commit(cb);

            hook(macs::error_code());
        }
    }

private:
    void moveMailishMessages() {
        const auto q = query<query::MailishMoveMessages>(query::SrcFolderId(srcFid), query::DstFolderId(dstFid), moveCoords);
        transaction->execute(q, [this, self=this->shared_from_this()](error_code err) {
            if (err) {
                hook(std::move(err));
            } else {
                (*this)();
            }
        });
    }

    void moveMessages() {
        moveFunctor_(handler());
    }

    void checkError(error_code err) {
        if (err) {
            hook(std::move(err));
        } else {
            (*this)();
        }
    }

    OnUpdate handler() {
        return [this, self=this->shared_from_this()](error_code err, Revision /*r*/) {
            if (err) {
                hook(std::move(err));
            } else {
                (*this)();
            }
        };
    }

    template <typename T, typename ...Args >
    T query( Args&& ... args) const {
        return queryRepository->query<T>(query::UserId(uid), std::forward<Args>(args)...);
    }

    ConnProvider connProvider;
    TransactionPtr transaction;
    pgg::query::RepositoryPtr queryRepository;
    pgg::RequestInfo requestInfo;
    OnExecute hook;
    std::string uid;
    std::string srcFid;
    std::string dstFid;
    MailishMoveCoordsChunk moveCoords;
    pgg::Milliseconds transactionTimeout;
    std::function<void(OnUpdate)> moveFunctor_;
};

} // namespace pg
} // namespace macs

#include <boost/asio/unyield.hpp>
