#pragma once

#include <pgg/query/repository.h>
#include <internal/query/comment.h>

#include <macs_pg/mailish/repository.h>

#include <macs/folders_repository.h>
#include <macs/labels_repository.h>
#include <macs/tabs_repository.h>

namespace macs {
namespace pg {

template <typename DatabaseGenerator>
class MailishRepository: public macs::MailishRepository {
public:
    MailishRepository(const DatabaseGenerator& generator, const std::string& uid, pgg::query::RepositoryPtr queryRepository,
            const pgg::RequestInfo& requestInfo, FoldersRepositoryPtr foldersRepository, LabelsRepositoryPtr labelsRepository,
            TabsRepositoryPtr tabsRepository, pgg::Milliseconds transactionTimeout)
        : uid_(uid)
        , db(generator)
        , queryRepository_(queryRepository)
        , foldersRepository_(foldersRepository)
        , labelsRepository_(labelsRepository)
        , tabsRepository_(tabsRepository)
        , transactionTimeout_(transactionTimeout)
        , requestInfo_(requestInfo)
    {
    }

private:
    void asyncMoveMessages(const std::string& srcFid,
                           const std::string& dstFid,
                           const std::optional<Tab::Type>& dstTab,
                           const MailishMoveCoordsChunk& chunk,
                           OnExecute hook) const override;
    void asyncDeleteMailishEntries(const std::string& fid, const ImapIds& imapIds, OnExecute hook) const override;
    void asyncIncrementErrors(const std::string& fid, const uint64_t imapId, const uint32_t errors, OnExecute hook) const override;
    void asyncSaveAccount(const MailishAuthData& data, const MailishAccount& account, OnExecute hook) const override;
    void asyncInvalidateAuthData(const std::string& tokenId, OnExecute hook) const override;
    void asyncGetAuthData(OnMailishAuthData hook) const override;
    void asyncGetAccount(OnMailishAccount hook) const override;
    void asyncGetFolders(OnMailishFolders hook) const override;
    void asyncUpdateDownloadedRange(const std::string& fid, const uint32_t begin, const uint32_t end, OnExecute hook) const override;
    void asyncGetMessages(const std::string& fid, const uint32_t highest_imap_id, const uint32_t count, OnMailishMessages hook) const override;
    void asyncGetMessages(const std::string& fid, const std::vector<uint64_t>& imap_ids, OnMailishMessages hook) const override;
    void asyncGetMessages(const std::vector<std::string>& mids, OnMailishMessages hook) const override;
    void asyncGetNotDownloadedMessages(const uint32_t count, OnMailishMessages hook) const override;
    void asyncGetMessagesTop(const std::string& fid, const uint32_t count, OnMailishMessages hook) const override;
    void asyncEraseSecurityLocks(OnExecute hook) const override;
    void asyncInitFolder(const std::string& fid, const MailishFolderInfo& info, OnExecute hook) const override;
    void asyncUpdateFolder(const Folder& folder, const MailishFolderInfo& info, OnExecute hook) const override;
    void asyncDeleteMailishFolderEntries(const std::vector<Fid>& fids, OnExecute hook) const override;
    void asyncCreateFolder(const std::string& name, const std::string& parentId, const Folder::Symbol& symbol,
        const MailishFolderInfo& info, OnMailishFolder hook) const override;
    void asyncUpdateLastSyncTs(const std::time_t lastSyncTs, OnExecute hook) const override;
    void asyncStore(const Envelope& envelope, const MimeParts& mime, const ThreadMeta& meta, const NotificationMode& notificationMode,
        const MailishMessageInfo& mailishInfo, OnUpdateEnvelope hook) const override;

    const std::string & uid() const { return uid_; }

    template <typename T, typename ...ArgsT >
    T query(ArgsT&& ... args) const {
        return makeQueryWithComment<T>(queryRepository(), uid(),
                std::forward<ArgsT>(args) ...);
    }

    const pgg::query::Repository & queryRepository() const {
        return *queryRepository_;
    }

    const std::string uid_;
    const DatabaseGenerator db;
    pgg::query::RepositoryPtr queryRepository_;
    FoldersRepositoryPtr foldersRepository_;
    LabelsRepositoryPtr labelsRepository_;
    TabsRepositoryPtr tabsRepository_;
    pgg::Milliseconds transactionTimeout_;
    pgg::RequestInfo requestInfo_;
};

template <typename DatabaseGenerator>
MailishRepositoryPtr createMailishRepository(
    DatabaseGenerator dbg,
    const std::string& uid,
    pgg::query::RepositoryPtr repo,
    const pgg::RequestInfo& requestInfo,
    FoldersRepositoryPtr foldersRepository,
    LabelsRepositoryPtr labelsRepository,
    TabsRepositoryPtr tabsRepository,
    pgg::Milliseconds timeout);

}; // namespace pg
}; // namespace macs

#include "repository.ipp"
