#include <macs_pg/mailish/auth_data_factory.h>
#include <macs_pg/mailish/folder_factory.h>
#include <macs_pg/mailish/message_factory.h>
#include <pgg/database/database.h>
#include <pgg/hooks/wrap.h>
#include <internal/mailish/repository.h>
#include <internal/mailish/query.h>
#include <internal/mailish/account.h>
#include <internal/mailish/update_folder.h>
#include <internal/mailish/create_folder.h>
#include <internal/mailish/move_messages.h>
#include <internal/mailish/store_message.h>
#include <internal/reflection/mailish_auth_data.h>
#include <internal/reflection/mailish_folder.h>
#include <internal/reflection/mailish_message.h>

namespace macs {
namespace pg {

namespace detail {

inline MailishMessage toMailishMessage(const reflection::MailishMessage& v) {
    MailishMessageFactory factory;
    factory
        .uid(std::to_string(v.uid))
        .externalImapId(std::to_string(v.external_imap_id))
        .fid(std::to_string(v.fid))
        .mid(std::to_string(v.mid))
        .externalReceivedDate(v.external_received_date)
        .errors(v.errors);
    return factory.release();
}

inline MailishAuthData toMailishAuthData(const reflection::MailishAuthData& v) {
    MailishAuthDataFactory factory;
    factory
        .tokenId(v.token_id)
        .authType(AuthType::fromString(v.auth_type))
        .uuid(v.uuid)
        .lockFlag(v.lock_flag)
        .lastValid(v.last_valid)
        .oauthApplication(v.oauth_app)
        .imapCredentials(v.imap_credentials)
        .smtpCredentials(v.smtp_credentials);
    return factory.release();
}

inline MailishFolder toMailishFolder(const reflection::MailishFolder& v) {
    MailishFolderFactory factory;
    factory
        .fid(std::to_string(v.fid))
        .path(v.imap_path)
        .uidValidity(static_cast<uint32_t>(v.uidvalidity))
        .downloadedRangeBegin(static_cast<uint32_t>(v.range_start))
        .downloadedRangeEnd(static_cast<uint32_t>(v.range_end));
    return factory.release();
}

} // namespace detail

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncMoveMessages(const std::string& srcFid,
                                          const std::string& dstFid,
                                          const std::optional<Tab::Type>& dstTab,
                                          const MailishMoveCoordsChunk& chunk,
                                          OnExecute hook) const {
    foldersRepository_->getAllFolders([this, self = this->shared_from_this(), srcFid, dstFid, dstTab, chunk, hook = std::move(hook)]
                                      (error_code err, FolderSet f) {
        if (err) {
            return hook(std::move(err));
        }

        auto inboxFid = f.fid(Folder::Symbol::inbox);
        tabsRepository_->getAllTabs([this, self, srcFid = std::move(srcFid), dstFid = std::move(dstFid), dstTab = std::move(dstTab), chunk = std::move(chunk), inboxFid = std::move(inboxFid), hook = std::move(hook)]
                                    (error_code err, TabSet tabs) {
            if (err) {
                return hook(std::move(err));
            }

            using Transaction = pgg::query::fallback::Transaction<DatabaseGenerator, pgg::database::fallback::rules::Master>;
            auto transaction = std::make_shared<Transaction>(queryRepository_);
            auto move = std::make_shared<MoveMailishMessages<decltype(db), decltype(transaction)>>(
                    db, transaction, queryRepository_, requestInfo_, std::move(hook), uid(),
                    srcFid, dstFid, dstTab, chunk, inboxFid, tabs,
                    transactionTimeout_);
            (*move)();
        });
    });
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncDeleteMailishEntries(const std::string& fid, const ImapIds& imapIds, OnExecute hook) const {
    const auto q = query<query::MailishDeleteEntries>(query::FolderId(fid), query::ImapIds(imapIds));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncIncrementErrors(const std::string& fid, const uint64_t imapId, const uint32_t errors, OnExecute hook) const {
    const auto q = query<query::MailishIncrementErrors>(query::FolderId(fid), query::ImapMessageId(std::to_string(imapId)), query::ErrorsCount(errors));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncSaveAccount(const MailishAuthData& data, const MailishAccount& account, OnExecute hook) const {
    const auto q = query<query::SaveMailishAccount>(data, account);
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncInvalidateAuthData(const std::string& tokenId, OnExecute hook) const {
    const auto q = query<query::InvalidateMailishAuthData>(query::TokenId(tokenId));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetAuthData(OnMailishAuthData hook) const {
    const auto q = query<query::GetMailishAuthData>();
    auto wrapper = pgg::wrapHook<reflection::MailishAuthData>(std::move(hook), detail::toMailishAuthData);
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetAccount(OnMailishAccount hook) const {
    const auto q = query<query::GetMailishAccount>();
    auto wrapper = pgg::wrapHook<reflection::MailishAccount>(std::move(hook), toMailishAccount);
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetFolders(OnMailishFolders hook) const {
    const auto q = query<query::GetMailishFolders>();
    auto wrapper = pgg::wrapHook<reflection::MailishFolder>(std::move(hook), detail::toMailishFolder);
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncUpdateDownloadedRange(const std::string& fid, const uint32_t begin, const uint32_t end, OnExecute hook) const {
    const auto q = query<query::MailishUpdateDownloadedRange>(query::FolderId(fid), query::ImapIdFrom(begin), query::ImapIdTo(end));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetMessages(const std::string& fid, const uint32_t highest_imap_id, const uint32_t count, OnMailishMessages hook) const {
    const auto q = query<query::GetMailishMessagesChunkById>(query::FolderId(fid), query::ImapIdFrom(highest_imap_id), query::MessagesCount(count));
    auto wrapper = pgg::wrapHook<reflection::MailishMessage>(std::move(hook), detail::toMailishMessage);
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetMessages(const std::string& fid, const std::vector<uint64_t>& imap_ids, OnMailishMessages hook) const {
    const auto q = query<query::GetMailishMessagesByImapIds>(query::FolderId(fid), query::ImapIds(imap_ids));
    auto wrapper = pgg::wrapHook<reflection::MailishMessage>(std::move(hook), detail::toMailishMessage);
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetMessages(const std::vector<std::string>& mids, OnMailishMessages hook) const {
    const auto q = query<query::GetMailishMessagesByMids>(query::MailIdVec(mids));
    auto wrapper = pgg::wrapHook<reflection::MailishMessage>(std::move(hook), detail::toMailishMessage);
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetNotDownloadedMessages(const uint32_t count, OnMailishMessages hook) const {
    const auto q = query<query::GetNotDownloadedMailishMessages>(query::MessagesCount(count));
    auto wrapper = pgg::wrapHook<reflection::MailishMessage>(std::move(hook), detail::toMailishMessage);
    db()->fetch(q, std::move(wrapper));
};


template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncGetMessagesTop(const std::string& fid, const uint32_t count, OnMailishMessages hook) const {
    const auto q = query<query::GetMailishMessagesByTop>(query::FolderId(fid), query::MessagesCount(count));
    auto wrapper = pgg::wrapHook<reflection::MailishMessage>(std::move(hook), detail::toMailishMessage);
    db()->fetch(q, std::move(wrapper));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncEraseSecurityLocks(OnExecute hook) const {
    const auto q = query<query::MailishEraseSecurityLocks>();
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncInitFolder(const std::string& fid, const MailishFolderInfo& info, OnExecute hook) const {
    const auto q = query<query::MailishInitFolder>(query::FolderId(fid), info);
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncUpdateFolder(const Folder& folder, const MailishFolderInfo& info, OnExecute hook) const {
    foldersRepository_->getAllFolders([this, self = this->shared_from_this(), folder, info, hook]
            (error_code err, FolderSet folders) mutable
    {
        if (err) {
            hook(std::move(err));
        } else {
            auto updateFolder = mailish::UpdateFolder(queryRepository_, foldersRepository_, std::move(hook), uid_,
                    folder, info, transactionTimeout_, std::move(folders));
            pgg::query::fallback::runTransactional(db, queryRepository_, std::move(updateFolder));
        }
    });
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncDeleteMailishFolderEntries(const std::vector<Fid>& fids, OnExecute hook) const {
    const auto q = query<query::MailishDeleteFolderEntries>(query::FolderIdVector(fids));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncCreateFolder(const std::string& name, const std::string& parentId, const Folder::Symbol& symbol,
    const MailishFolderInfo& info, OnMailishFolder hook) const
{
    foldersRepository_->getAllFolders([this, self = this->shared_from_this(), name, parentId, symbol, info, hook]
            (error_code err, FolderSet folders) mutable
    {
        if (err) {
            hook(std::move(err));
        } else {
            auto createFolder = CreateMailishFolder(queryRepository_, foldersRepository_, std::move(hook), uid_, name,
                    parentId, symbol, info, transactionTimeout_, std::move(folders));
            pgg::query::fallback::runTransactional(db, queryRepository_, std::move(createFolder));
        }
    });
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncUpdateLastSyncTs(const std::time_t lastSyncTs, OnExecute hook) const {
    const auto q = query<query::UpdateMailishLastSyncTs>(query::LastSyncTs(lastSyncTs));
    db()->execute(q, std::move(hook));
}

template <typename DatabaseGenerator>
void MailishRepository<DatabaseGenerator>::asyncStore(const Envelope& envelope, const MimeParts& mime, const ThreadMeta& meta, const NotificationMode& notificationMode,
    const MailishMessageInfo& mailishInfo, OnUpdateEnvelope hook) const
{
    using Transaction = pgg::query::fallback::Transaction<DatabaseGenerator, pgg::database::fallback::rules::Master>;

    labelsRepository_->getAllLabels([this, self = this->shared_from_this(), envelope, mime, meta, notificationMode,
        mailishInfo, hook = std::move(hook)]
        (error_code err, LabelSet labels) mutable
    {
        if (err) {
            hook(std::move(err));
        } else {
            foldersRepository_->getAllFolders([this, self, envelope, mime, meta, notificationMode,
                mailishInfo, labels = std::move(labels), hook]
                (error_code err, FolderSet folders) mutable
            {
                if (err) {
                    hook(std::move(err));
                } else {
                    auto transaction = boost::make_shared<Transaction>(queryRepository_);
                    auto storeMailishMessage = boost::make_shared<StoreMailishMessage<decltype(db), decltype(transaction)>>(
                            db, transaction, queryRepository_, uid(), requestInfo_,
                            envelope, mime, meta, std::move(labels), std::move(folders), notificationMode,
                            mailishInfo, std::move(hook), transactionTimeout_);
                    (*storeMailishMessage)();
                }
            });
        }
    });
}

} // namespace pg
} // namespace macs
